
#ifdef _UNICODE
#define UNICODE
#endif

#include <windows.h>
#include <process.h>
#include <stdio.h>
#include <stdlib.h>
#include <tchar.h>

#include <list>
#include <vector>
#include <set>

#define countof(array) (sizeof(array) / sizeof(array[0]))

/* I had trouble with std::basic_string<wchar_t>, so use special String */
class String {
    TCHAR* m_buffer;
    int m_length;
    template<class T>
    void initialize(T string, int length) {
        m_buffer = new TCHAR[length + 1];
        m_length = length;
        for (int i = 0; i < length; i++)
            m_buffer[i] = *string++;
        m_buffer[m_length] = _T('\0');
    }
    String(const String& str1, const String& str2) {
        int len1 = str1.length();
        int len2 = str2.length();
        
        m_length = len1 + len2;
        m_buffer = new TCHAR[m_length + 1];
        memcpy(m_buffer, (LPCTSTR) str1, sizeof(TCHAR) * len1);
        memcpy(m_buffer + len1, (LPCTSTR) str2, sizeof(TCHAR) * len2);
        m_buffer[m_length] = _T('\0');
    }
    
public:
    String(const String& string) {
        initialize((LPCTSTR) string, string.length());
    }
    String& operator= (const String& string) {
        delete[] m_buffer;
        initialize((LPCTSTR) string, string.length());
        return *this;
    }
    
    ~String() {
        delete[] m_buffer;
    }
    String() {
        initialize(_T(""), 0);
    }
    String(LPCTSTR string) {
        initialize(string, _tcslen(string));
    }
    template<class T>
    String(T string, int length) {
        initialize(string, length);
    }
    
    
    
    operator LPCTSTR () const {
        return m_buffer;
    }
    
    int length() const {
        return m_length;
    }
    
    int indexOf(TCHAR ch, int fromIndex) const {
        TCHAR* tail = _tcschr(m_buffer + fromIndex, ch);
        if (tail == NULL)
            return -1;;
        return tail - m_buffer;
    }
    int indexOf(TCHAR ch) const {
        return indexOf(ch, 0);
    }
    int indexOf(String str, int fromIndex) const {
        TCHAR* tail = _tcsstr(m_buffer + fromIndex, str);
        if (tail == NULL)
            return -1;;
        return tail - m_buffer;
    }
    int indexOf(String str) const {
        return indexOf(str, 0);
    }
    
    int lastIndexOf(TCHAR ch, int fromIndex) const {
        TCHAR* tail = _tcsrchr(m_buffer + fromIndex, ch);
        if (tail == NULL)
            return -1;;
        return tail - m_buffer;
    }
    int lastIndexOf(TCHAR ch) const {
        return lastIndexOf(ch, 0);
    }
    
    String substring(int begin, int end) const {
        return String(m_buffer + begin, end - begin);
    }
    
    String replace(String oldString, String newString) const {
        int index = indexOf(oldString);
        if (index < 0)
            return *this;
        
        String result = substring(0, index);
        for (;;) {
            result = result + newString;
            index += oldString.length();
            int nextIndex = indexOf(oldString, index);
            if (nextIndex < 0) {
                break;
            }
            result = result + substring(index, nextIndex);
            index = nextIndex;
        }
        return result + substring(index, length());
    }
    
    String toUpperCase() const {
        String result = *this;
        _tcsupr(result.m_buffer);
        return result;
    }
    
    String operator+ (const String& rhs) const {
        return String(*this, rhs);
    }
    String operator+ (LPCTSTR rhs) const {
        return String(*this, String(rhs));
    }
};
String operator+ (LPCTSTR lhs, const String& rhs) {
    return String(lhs) + rhs;
}

typedef std::vector<TCHAR> StringBuffer;
void append(StringBuffer& buffer, const String& str) {
    LPCTSTR p = (LPCTSTR) str;
    for (int i = 0; i < str.length(); i++)
        buffer.push_back(*p++);
}

typedef std::list<String> StringList;
#ifdef VERBOSE
void printStringList(const StringList& list) {
    StringList::const_iterator it = list.begin();
    for (; it != list.end(); it++) {
        String token = *it;
        _tprintf(_T("verbose:     %s\n"), (LPCTSTR) token);
    }
}
#endif

String getModuleFileName() {
    TCHAR name[MAX_PATH + 1];
    int length = ::GetModuleFileName(NULL, name, MAX_PATH);
    return String(name, length);
}

String getSettingFileName(const String& moduleName) {
    int index = moduleName.lastIndexOf(_T('\\'));
    if (index >= 0)
        index = moduleName.lastIndexOf(_T('.'), index);
    if (index >= 0)
        return moduleName.substring(0, index) + _T(".notavaboot");
    else
        return moduleName + _T(".notavaboot");
}

String getParent(const String& filename) {
    int index = filename.lastIndexOf(_T('\\'));
    if (index >= 0)
        return filename.substring(0, index + 1);
    index = filename.lastIndexOf(_T(':'));
    if (index >= 0)
        return filename.substring(0, index + 1);
    return String(_T(""));
}
String getTitle(const String& filename) {
    int index = filename.lastIndexOf(_T('\\'));
    String title = filename.substring(index + 1, filename.length());
    index = title.lastIndexOf(_T('.'));
    if (index >= 0)
        return title.substring(0, index);
    else
        return title;
}

bool isFileAvailable(String fileName) {
    return ::GetFileAttributes((LPCTSTR) fileName) != 0xFFFFFFFF;
}

String getEnvironment(String key) {
    LPCTSTR value = _tgetenv(key);
    if (value != 0) {
#ifdef VERBOSE
        _tprintf(_T("verbose: environment %s = %s\n"), (LPCTSTR) key, (LPCTSTR) value);
#endif
        return String(value);
    } else {
#ifdef VERBOSE
        _tprintf(_T("verbose: environment %s not found.\n"), (LPCTSTR) key);
#endif
        return String();
    }
}

String getCommandLine() {
    LPCTSTR p = ::GetCommandLine();
#ifdef VERBOSE
    _tprintf(_T("verbose: original command line: %s\n"), (LPCTSTR) p);
#endif
    
    bool quoted = false;
    for (; *p != _T('\0'); p += _tclen(p)) {
        if (!quoted && _istspace(*p))
            break;
        if (*p == _T('\"'))
            quoted = !quoted;
    }
    for (; *p != _T('\0'); p += _tclen(p)) {
        if (!_istspace(*p))
            break;
    }
    return String(p);
}

StringList parseText(String text) {
    StringList result;
    
    LPCTSTR p = (LPCTSTR) text;
    LPCTSTR q = p;
    bool quoted = false;
    while (*p != _T('\0')) {
        TCHAR ch = (TCHAR) *p;
        if (!quoted && _istspace(ch)) {
            if (q < p)
                result.push_back(String(q, p - q));
            p += _tclen(p);
            q = p;
        } else {
            if (ch == _T('"'))
                quoted = !quoted;
            p += _tclen(p);
        }
    }
    if (q < p)
        result.push_back(String(q, p - q));
    return result;
}
StringList parseFile(String fileName) {
#ifdef VERBOSE
    _tprintf(_T("verbose: reading %s\n"), (LPCTSTR) fileName);
#endif
    
    FILE* fp = _tfopen(fileName, _T("rt"));
    if (fp == 0) {
        _ftprintf(stderr, _T("error: cannot open %s\n"), (LPCTSTR) fileName);
        exit(-1);
    }
    
    StringBuffer buffer;
    for (;;) {
        _TINT ch = _fgettc(fp);
        if (ch == _TEOF)
            break;
        buffer.push_back((TCHAR) ch);
    }
    fclose(fp);
    return parseText(String(buffer.begin(), buffer.size()));
}

StringList parsePath(String text, bool& containsSemicolon) {
    StringList result;
    containsSemicolon = false;
    
    LPCTSTR p = (LPCTSTR) text;
    LPCTSTR q = p;
    bool quoted = false;
    while (*p != _T('\0')) {
        TCHAR ch = (TCHAR) *p;
        if (!quoted && ch == _T(';')) {
            containsSemicolon = true;
            if (q < p)
                result.push_back(String(q, p - q));
            p += _tclen(p);
            q = p;
        } else {
            if (ch == _T('"'))
                quoted = !quoted;
            p += _tclen(p);
        }
    }
    if (q < p)
        result.push_back(String(q, p - q));
    return result;
}

String expandEnvironment(String text) {
    if (text.indexOf(_T('%')) == 0 && text.indexOf(_T('%'), 1) == text.length() - 1 && text.length() > 2) {
        String value = getEnvironment(text.substring(1, text.length() - 1));
        if (isFileAvailable(value))
            value = _T("\"") + value + _T("\"");
        return value;
    }
    
    StringBuffer buffer;
    LPCTSTR p = (LPCTSTR) text;
    bool quoted = false;
    while (*p != _T('\0')) {
        TCHAR ch = (TCHAR) *p;
        if (quoted || ch != _T('%')) {
            int size = _tclen(p);
            for (int i = 0; i < size; i++)
                buffer.push_back(*p++);
            if (ch == _T('"'))
                quoted = !quoted;
        } else {
            p += _tclen(p);
            ch = (TCHAR) *p;
            if (ch == _T('%')) {
                buffer.push_back(ch);
                p += _tclen(p);
            } else if (ch == _T('*')) {
                append(buffer, getCommandLine());
                p += _tclen(p);
            } else if (ch == _T('@')) {
                String base = _T("\"") + getParent(getModuleFileName()) + _T("\"");
                append(buffer, base);
                p += _tclen(p);
            } else {
                LPCTSTR q = p;
                while (*p != _T('\0') && *p != _T('%')) {
                    TCHAR ch = (TCHAR) *p;
                    p += _tclen(p);
                }
                if (*p == _T('%')) {
                    String value = getEnvironment(String(q, p - q));
                    value = _T("\"") + value + _T("\"");
                    append(buffer, value);
                    p += _tclen(p);
                } else {
                    buffer.push_back(_T('%'));
                    while (q != p) {
                        buffer.push_back(*q++);
                    }
                }
            }
        }
    }
    return String(buffer.begin(), buffer.size());
}

String removeQuotes(String text, bool& wildcard) {
    StringBuffer result;
    wildcard = false;
    
    LPCTSTR p = (LPCTSTR) text;
    bool quoted = false;
    while (*p != _T('\0')) {
        TCHAR ch = *p;
        if (ch == _T('"'))  {
            quoted = !quoted;
            p += _tclen(p);
        } else {
            if (!quoted && (ch == _T('?') || ch == _T('*')))
                wildcard = true;
            int len = _tclen(p);
            for (int i = 0; i < len; i++) {
                result.push_back(*p++);
            }
        }
    }
    return String(result.begin(), result.size());
}
String expandWildcard(String text) {
    bool semicolon = false;
    StringList list = parsePath(text, semicolon);
    
    StringList result;
    StringList::const_iterator it = list.begin();
    for (; it != list.end(); it++) {
        String token = *it;
        bool wildcard = false;
        token = removeQuotes(token, wildcard);
        if (!wildcard) {
            result.push_back(token);
        } else {
#ifdef VERBOSE
            _tprintf(_T("verbose: pattern match: %s\n"), (LPCTSTR) token);
#endif
            WIN32_FIND_DATA find;
            HANDLE h = ::FindFirstFile(token, &find);
            if (h != INVALID_HANDLE_VALUE) {
                String base = getParent(token);
                do {
                    if (find.cFileName[0] != _T('.')) {
                        String file = base + find.cFileName;
                        result.push_back(file);
#ifdef VERBOSE
                        _tprintf(_T("verbose:     %s\n"), (LPCTSTR) file);
#endif
                    }
                } while (::FindNextFile(h, &find));
                ::FindClose(h);
            }
        }
    }
    
    if (semicolon) {
        StringBuffer r;
        append(r, _T("\""));
        bool first = true;
        StringList::const_iterator it = result.begin();
        for (; it != result.end(); it++) {
            String token = *it;
            if (first) {
                append(r, token);
                first = false;
            } else {
                append(r, _T(";"));
                append(r, token);
            }
        }
        append(r, _T("\""));
        return String(r.begin(), r.size());
    } else {
        StringBuffer r;
        bool first = true;
        StringList::const_iterator it = result.begin();
        for (; it != result.end(); it++) {
            String token = *it;
            if (first) {
                first = false;
            } else {
                append(r, _T(" "));
            }
            append(r, _T("\""));
            append(r, token);
            append(r, _T("\""));
        }
        return String(r.begin(), r.size());
    }
}



LPTSTR searchPaths[] = {
    _T("..\\lib"),
    _T(""),
    _T("lib"),
    _T(".."),
};

int bootJar(const String& moduleFileName) {
    const String base = getParent(moduleFileName);
    const String title = getTitle(moduleFileName);
    const String jarName = title + _T(".jar");
    String jarFile;
    int i = 0;
    for (; i < countof(searchPaths); i++) {
        jarFile = base + searchPaths[i] + _T("\\") + jarName;
#ifdef VERBOSE
        _tprintf(_T("verbose: check %s\n"), (LPCTSTR) jarFile);
#endif
        if (isFileAvailable(jarFile))
            break;
    }
    if (i >= countof(searchPaths)) {
        _ftprintf(stderr, _T("error: %s not found.\n"), (LPCTSTR) jarName);
        return -1;
    }
    
    String commandLine = _T("java ");
    commandLine = commandLine + getEnvironment(title.toUpperCase() + _T("_VMOPTIONS"));
    commandLine = commandLine + _T(" -jar \"") + jarFile + _T("\" ") + getCommandLine();
#ifdef VERBOSE
    _tprintf(_T("verbose: booting %s\n"), (LPCTSTR) commandLine);
#endif
    return _tsystem(commandLine);
}



int _tmain(int argc, TCHAR **argv) {
    const String moduleFileName = getModuleFileName();
    const String settingFileName = getSettingFileName(moduleFileName);
    
#ifdef VERBOSE
    _tprintf(_T("verbose: I am %s\n"), (LPCTSTR) moduleFileName);
#endif
    
    if (!isFileAvailable(settingFileName)) {
#ifdef VERBOSE
        _tprintf(_T("verbose: %s not found.\n"), (LPCTSTR) settingFileName);
#endif
        return bootJar(moduleFileName);
    }
    
    StringList tokens = parseFile(settingFileName);
#ifdef VERBOSE
    printStringList(tokens);
#endif
    
    StringList tokens2;
    StringList::const_iterator it = tokens.begin();
    for (; it != tokens.end(); it++) {
        String token = *it;
        StringList expanded = parseText(expandEnvironment(token));
        tokens2.insert(tokens2.end(), expanded.begin(), expanded.end());
    }
    
#ifdef VERBOSE
    _tprintf(_T("verbose: environment expanded.\n"));
    printStringList(tokens2);
#endif
    
    StringList tokens3;
    it = tokens2.begin();
    for (; it != tokens2.end(); it++) {
        String token = *it;
        StringList expanded = parseText(expandWildcard(token));
        tokens3.insert(tokens3.end(), expanded.begin(), expanded.end());
    }
    
#ifdef VERBOSE
    _tprintf(_T("verbose: wildcard expanded.\n"));
    printStringList(tokens3);
#endif
    
    StringBuffer result;
    bool first = true;
    it = tokens3.begin();
    for (; it != tokens3.end(); it++) {
        String token = *it;
        if (first) {
            first = false;
            append(result, token.replace(_T("\""), _T("")));
        } else {
            append(result, _T(" "));
            append(result, token);
        }
    }
    String commandLine(result.begin(), result.size());
#ifdef VERBOSE
    _tprintf(_T("verbose: booting %s\n"), (LPCTSTR) commandLine);
#endif
    return _tsystem(commandLine);
}
