
package jp.gr.java_conf.koto.notavacc.parser;

import java.io.*;
import java.util.*;
import jp.gr.java_conf.koto.notavacc.*;

public class Parser extends Original {
    protected Node createNode(int symbolID, NodeInitializationParameters parameters) {
        switch (symbolID) {
        case Root.ID:
            return new Root(logger, parameters);
        case InlineExpression.ID:
            return new InlineExpressionReplacement(parameters);
        }
        return super.createNode(symbolID, parameters, true);
    }
    protected void taggedTokenShifted(Token token, int tag) {
        switch (tag) {
        case TAG_OBSOLETE_WHITE_TOKEN:
            logger.warn("{0} is obsolete.  Use ''$white $token IDENTIFIER = ... ;''", token);
            break;
        default:
            assert false;
        }
    }
    
    protected LexicalAnalyzer createLexicalAnalyzer(String sourceName, CharSequence text, int tabStop) throws ParseException {
        return new Default.LexicalAnalyzer(sourceName, text, tabStop) {
            protected char nextChar() throws ParseException {
                return super.nextUnicodeEscapedChar();
            }
            public Token next() throws ParseException {
                Token result = super.next();
                if (result.getSymbolID() == TOKEN_IDENTIFIER) {
                    String image = result.getImage();
                    int i = 0;
                    for (; i < image.length(); i++) {
                        char ch = image.charAt(i);
                        if (i == 0) {
                            if (!Character.isJavaIdentifierStart(ch))
                                break;
                        } else {
                            if (!Character.isJavaIdentifierPart(ch))
                                break;
                        }
                    }
                    if (i < image.length())
                        throw new ParseException("The " + i + "th letter \\u" + Integer.toHexString(image.charAt(i)) +  " of the token " + result + " cannot be used.", result);
                }
                return result;
            }
        };
    }
    
    private Logger logger;
    public Parser(Logger logger) {
        this.logger = logger;
    }
    
    public static class Root extends Default.Root {
        private Logger logger;
        public Root(Logger logger, NodeInitializationParameters parameters) {
            super(parameters, true);
            this.logger = logger;
            
            buildDefinitionMap();
            bindNames();
            buildSetOfDefinitionsHavingExplicitParameterLabel();
        }
        
        private String packageName = null;
        public String getPackageName() {
            if (packageName != null)
                return packageName;
            
            PackageOption po = packageOption();
            if (po == null)
                return null;
            
            StringBuffer result = new StringBuffer();
            Iterator it0 = po.value().identifiers().iterator();
            while (it0.hasNext()) {
                Token identifier = (Token) it0.next();
                result.append(identifier.getImage());
                if (it0.hasNext())
                    result.append(".");
            }
            
            packageName = result.toString();
            return packageName;
        }
        
        public boolean hasProtectedConstructor() {
            return constructorScope() != null;
        }
        
        private List typeDefinitions = null;
        public List typeDefinitions() {
            if (typeDefinitions == null) {
                typeDefinitions = new LinkedList();
                accept(new Visitor() {
                    public void visit(TypeDefinition def) {
                        typeDefinitions.add(def);
                    }
                });
            }
            return typeDefinitions;
        }
        
        public String getCanonicalName(Definition definition) {
            return definition.identifier().getImage();
        }
        
        private Map canonicalNameToDefinition;
        private void buildDefinitionMap() {
            canonicalNameToDefinition = new LinkedHashMap();
            final Map duplexNames = new LinkedHashMap();
            accept(new Visitor() {
                public void visit(Definition def) {
                    String canonicalName = getCanonicalName(def);
                    Definition olddef = (Definition) canonicalNameToDefinition.put(canonicalName, def);
                    if (olddef != null) {
                        List definitions = (List) duplexNames.get(canonicalName);
                        if (definitions == null) {
                            definitions = new LinkedList();
                            duplexNames.put(canonicalName, definitions);
                            definitions.add(olddef);
                        }
                        definitions.add(def);
                    }
                }
            });
            if (!duplexNames.isEmpty()) {
                Iterator it = duplexNames.values().iterator();
                while (it.hasNext()) {
                    List list = (List) it.next();
                    Iterator it2 = list.iterator();
                    while (it2.hasNext()) {
                        Definition def = (Definition) it2.next();
                        logger.error("{0} defined twice or more.", def.identifier());
                    }
                }
            }
            if (logger.hasError())
                logger.fatal();
        }
        private Map nameToDefinition;
        private void bindNames() {
            nameToDefinition = new LinkedHashMap();
            accept(new Visitor() {
                public void visit(Name name) {
                    String image = name.identifier().getImage();
                    Definition def = (Definition) canonicalNameToDefinition.get(image);
                    nameToDefinition.put(name, def);
                    
                    if (def == null) {
                        logger.error("{0} is not defined.", name.identifier());
                    } else if (name instanceof TypeName) {
                        if (!(def instanceof TypeDefinition))
                            logger.error("{0} should be a type name.", name.identifier());
                    } else if (name instanceof SubtokenName) {
                        if (!(def instanceof SubtokenDefinition))
                            logger.error("{0} should be a token name or a subtoken name.", name.identifier());
                    } else {
                        assert name instanceof SymbolName;
                        boolean ok = false;
                        if (def instanceof TokenReservation) {
                            if (!(def instanceof WhiteTokenDefinition))
                                ok = true;
                        } else if (def instanceof TypeDefinition) {
                            ok = true;
                        } else if (def instanceof AliasDefinition) {
                            ok = true;
                        }
                        if (!ok)
                            logger.error("{0} should be a token/type/alias name.", name.identifier());
                    }
                }
            });
            if (logger.hasError())
                logger.fatal();
        }
        public Definition getDefinition(Name name) {
            Definition result = (Definition) nameToDefinition.get(name);
            assert result != null : "A definition missing for " + name.identifier();
            return result;
        }
        public TypeDefinition getDefinition(TypeName name) {
            return (TypeDefinition) getDefinition((Name) name);
        }
        public SubtokenDefinition getDefinition(SubtokenName name) {
            return (SubtokenDefinition) getDefinition((Name) name);
        }
        
        
        private Set definitionsHavingExplicitParameterLabel;
        private void buildSetOfDefinitionsHavingExplicitParameterLabel() {
            definitionsHavingExplicitParameterLabel = new LinkedHashSet();
            Iterator it = aliasDefinitions().iterator();
            while (it.hasNext()) {
                final AliasDefinition def = (AliasDefinition) it.next();
                def.accept(new Visitor() {
                    public void visit(LabeledExpression exp) {
                        if (exp.label().getImage().equals("$label"))
                            definitionsHavingExplicitParameterLabel.add(def);
                    }
                });
            }
        }
        public boolean hasExplicitParameterLabel(AliasDefinition definition) {
            return definitionsHavingExplicitParameterLabel.contains(definition);
        }
        
        
        public String getValue(StringLiteral literal) {
            StringBuffer result = new StringBuffer();
            String image = literal.token().getImage();
            int maz = image.length() - 1;
            for (int i = 1; i < maz;) {
                char ch = image.charAt(i++);
                if (ch != '\\') {
                    result.append(ch);
                } else {
                    assert i < maz;
                    ch = image.charAt(i++);
                    switch(ch) {
                    case 'n':   result.append('\n');    break;
                    case 't':   result.append('\t');    break;
                    case 'b':   result.append('\b');    break;
                    case 'r':   result.append('\r');    break;
                    case 'f':   result.append('\f');    break;
                    
                    case '\\':
                    case '\'':
                    case '\"':
                        result.append(ch);
                        break;
                    
                    /*case 'u':
                        while (ch == 'u') {
                            ch = image.charAt(i++);
                            assert i < maz;
                        }
                        assert Character.digit(ch, 16) != -1;
                        {
                            StringBuffer hex = new StringBuffer();
                            hex.append(ch);
                            int max = 4;
                            while (--max > 0) {
                                if (i >= maz)
                                    break;
                                ch = image.charAt(i);
                                if (Character.digit(ch, 16) == -1)
                                    break;
                                hex.append(ch);
                                i++;
                            }
                            result.append((char) Integer.parseInt(hex.toString(), 16));
                        }
                        break;*/
                    
                    default:
                        assert Character.digit(ch, 8) != -1;
                        {
                            StringBuffer octal = new StringBuffer();
                            octal.append(ch);
                            int max = ('0' <= ch && ch <= '3') ? 3 : 2;
                            while (--max > 0) {
                                if (i >= maz)
                                    break;
                                ch = image.charAt(i);
                                if (Character.digit(ch, 8) == -1)
                                    break;
                                octal.append(ch);
                                i++;
                            }
                            result.append((char) Integer.parseInt(octal.toString(), 8));
                        }
                        break;
                    }
                }
            }
            return result.toString();
        }
        
        public char getValue(CharacterLiteral literal) {
            String image = literal.token().getImage();
            char ch = image.charAt(1);
            if (ch != '\\') {
                return ch;
            }
            
            ch = image.charAt(2);
            switch(ch) {
            case 'n':   return '\n';
            case 't':   return '\t';
            case 'b':   return '\b';
            case 'r':   return '\r';
            case 'f':   return '\f';
            
            case '\\':
            case '\'':
            case '\"':
                return ch;
            
            /*case 'u':
                String hex = image.substring(3, image.length() - 1);
                return (char) Integer.parseInt(hex, 16);*/
            
            default:
                assert Character.digit(ch, 8) != -1;
                String octal = image.substring(2, image.length() - 1);
                return (char) Integer.parseInt(octal, 8);
            }
        }
    }
    
    private class InlineExpressionReplacement extends Default.InlineExpression {
        public InlineExpressionReplacement(NodeInitializationParameters parameters) {
            super(parameters, true);
            name = new SymbolNameReplacement();
            getChildNodes().add(0, name);
        }
        
        private final SymbolName name;
        public SymbolName name() {
            return name;
        }
        
        private class SymbolNameReplacement extends Default.Node implements SymbolName {
            public SymbolNameReplacement() {
                super(Collections.EMPTY_LIST);
            }
            public Original.Token identifier() {
                return typeDefinition().identifier();
            }
        }
    }
    
}

