
package jp.gr.java_conf.koto.notavacc.lrg;

import java.util.*;
import jp.gr.java_conf.koto.notavacc.*;
import jp.gr.java_conf.koto.notavacc.util.*;
import jp.gr.java_conf.koto.notavacc.dfa.*;
import jp.gr.java_conf.koto.notavacc.parser.Parser.*;
import jp.gr.java_conf.koto.notavacc.parser.Original.*;
import jp.gr.java_conf.koto.notavacc.parser.Parser.Root;

public final class LRTableGenerator extends Module {
    public LRTableGenerator(Environment env) {
        super(env);
    }
    
    private Root root;
    private DeterministicFiniteAutomaton dfa;
    public LRTable generate(Root root, DeterministicFiniteAutomaton dfa) {
        this.root = root;
        this.dfa = dfa;
        
        logger.verbose(environment.processingFile, "generating an extended CFG.");
        Map symbolToInitialState = new LinkedHashMap();
        {
            NondeterministicState finalState = new NondeterministicState();
            Iterator it = root.typeDefinitions().iterator();
            while (it.hasNext()) {
                TypeDefinition def = (TypeDefinition) it.next();
                if (def.abstractKeyword() == null) {
                    Symbol symbol = getNonterminal(def);
                    NondeterministicState initialState = new NondeterministicState();
                    link(initialState, def.expression(), finalState);
                    DeterministicConverter converter = new DeterministicConverter();
                    DeterministicState s = converter.createDeterministicState(Collections.singleton(initialState), finalState);
                    s = buildMinimalForm(s);
                    symbolToInitialState.put(symbol, s);
                }
            }
            it = root.aliasDefinitions().iterator();
            while (it.hasNext()) {
                AliasDefinition def = (AliasDefinition) it.next();
                TreeStackSet argumentLabels = TreeStackSet.EMPTY_SET;
                TreeStackSet labels = TreeStackSet.EMPTY_SET;
                if (root.hasExplicitParameterLabel(def)) {
                    argumentLabels = TreeStackSet.push(argumentLabels, "$label");
                } else {
                    labels = TreeStackSet.push(labels, "$label");
                }
                
                Symbol symbol = getNonterminal(def);
                NondeterministicState initialState = new NondeterministicState();
                link(false, TreeStackSet.EMPTY_SET, argumentLabels, labels, initialState, def.expression(), finalState);
                DeterministicConverter converter = new DeterministicConverter();
                DeterministicState s = converter.createDeterministicState(Collections.singleton(initialState), finalState);
                s = buildMinimalForm(s);
                symbolToInitialState.put(symbol, s);
            }
        }
        if (logger.hasError())
            logger.fatal();
        Set allStates = allStates(symbolToInitialState);
        logger.verbose(environment.processingFile, "generated.  {0} states.", new Integer(allStates.size()));
        //reportStates(symbolToInitialState, allStates);
        
        /*logger.verbose(environment.processingFile, "generating the minimal form.");
        symbolToInitialState = buildMinimalForm(symbolToInitialState, allStates);
        allStates = allStates(symbolToInitialState);
        logger.verbose(environment.processingFile, "generated.  {0} states.", new Integer(allStates.size()));
        reportStates(symbolToInitialState, allStates);*/
        
        logger.verbose(environment.processingFile, "generating a CFG.");
        symbolToInitialState = buildCFG(symbolToInitialState, allStates);
        allStates = allStates(symbolToInitialState);
        logger.verbose(environment.processingFile, "generated.  {0} states.", new Integer(allStates.size()));
        reportStates(symbolToInitialState, allStates);
        
        Map initialStateToSymbol = new HashMap(symbolToInitialState.size());
        Iterator it25 = symbolToInitialState.entrySet().iterator();
        while (it25.hasNext()) {
            Map.Entry entry = (Map.Entry) it25.next();
            initialStateToSymbol.put(entry.getValue(), entry.getKey());
        }
        assert initialStateToSymbol.size() == symbolToInitialState.size();
        
        logger.verbose(environment.processingFile, "generating LR states.");
        DeterministicState extendedFinalState = new DeterministicState();
        allStates.add(extendedFinalState);
        LRHelper lrHelper;
        {
            Map symbolToInitialLRState = new LinkedHashMap();
            LRStateBuilder builder = new LRStateBuilder(symbolToInitialState);
            Set definitions = new LinkedHashSet();
            Iterator it = root.typeDefinitions().iterator();
            while (it.hasNext()) {
                TypeDefinition def = (TypeDefinition) it.next();
                if (def.parsableKeyword() != null)
                    definitions.add(def);
            }
            
            Map kernelCoresToLRState = new LinkedHashMap();
            it = definitions.iterator();
            while (it.hasNext()) {
                TypeDefinition def = (TypeDefinition) it.next();
                TypeNonterminal symbol = getNonterminal(def);
                DeterministicState initialState = (DeterministicState) symbolToInitialState.get(symbol);
                
                DeterministicState extendedInitialState = new DeterministicState();
                DeterministicState extendedMediumState = new DeterministicState();
                
                extendedInitialState.linkToState.put(new Link(Collections.EMPTY_SET, symbol, def.identifier()), extendedMediumState);
                allStates.add(extendedInitialState);
                
                extendedMediumState.linkToState.put(new Link(Collections.EMPTY_SET, Terminal.EOF, def.identifier()), extendedFinalState);
                allStates.add(extendedMediumState);
                
                LRState state = builder.createLRState(Collections.singleton(extendedInitialState));
                symbolToInitialLRState.put(symbol, state);
            }
            LRState finalState = builder.createLRState(Collections.singleton(extendedFinalState));
            lrHelper = new LRHelper(symbolToInitialLRState, finalState);
        }
        logger.verbose(environment.processingFile, "generated.  {0} states.", new Integer(lrHelper.allLRStates().size()));
        //reportLRStates(lrHelper);
        
        logger.verbose(environment.processingFile, "tagging.");
        tagLRStates(lrHelper, initialStateToSymbol);
        logger.verbose(environment.processingFile, "tagged.");
        
        logger.verbose(environment.processingFile, "delivering look-ahead terminals.");
        {
            LookaheadDeliverer deliverer = new LookaheadDeliverer(symbolToInitialState, allStates);
            deliverer.deliverLookahead(lrHelper);
        }
        logger.verbose(environment.processingFile, "delivered.");
        
        {
            logger.verbose(environment.processingFile, "generating reductions.");
            ReductionGenerator generator = new ReductionGenerator(symbolToInitialState, initialStateToSymbol, allStates);
            Set conflictedStates = generator.generateReductions(lrHelper);
            logger.verbose(environment.processingFile, "generated.  {0} LALR(1)-conflicted state(s).", new Integer(conflictedStates.size()));
            reportLRStates(lrHelper);
            
            if (!conflictedStates.isEmpty()) {
                logger.verbose(environment.processingFile, "chasing conflicts.");
                ConflictChaser chaser = new ConflictChaser(lrHelper, generator, initialStateToSymbol);
                int count = chaser.chaseConflicts(conflictedStates);
                logger.verbose(environment.processingFile, "chased.  {0} maybe-conflicted state(s).", new Integer(count));
            }
        }
        
        logger.verbose(environment.processingFile, "generating LR table.");
        LRTable result = buildLRTable(lrHelper);
        logger.verbose(environment.processingFile, "generated.");
        
        return result;
    }
    
    
    private Map nonterminalToHint = new LinkedHashMap();
    private AliasNonterminal getNonterminal(AliasDefinition definition) {
        AliasNonterminal result = new AliasNonterminal(root.getCanonicalName(definition));
        nonterminalToHint.put(result, definition.identifier());
        return result;
    }
    private Token getHint(AliasNonterminal nonterminal) {
        return (Token) nonterminalToHint.get(nonterminal);
    }
    private TypeNonterminal getNonterminal(TypeDefinition definition) {
        TypeNonterminal result = new TypeNonterminal(root.getCanonicalName(definition));
        nonterminalToHint.put(result, definition.identifier());
        return result;
    }
    private Token getHint(TypeNonterminal nonterminal) {
        return (Token) nonterminalToHint.get(nonterminal);
    }
    private Token getHint(Nonterminal nonterminal) {
        if (nonterminal instanceof TypeNonterminal) {
            return getHint((TypeNonterminal) nonterminal);
        } else {
            return getHint((AliasNonterminal) nonterminal);
        }
    }
    
    
    
    private static class Link {
        public Set labels;
        public Symbol symbol;
        public Token hint;
        public Link(Set labels, Symbol symbol, Token hint) {
            this.labels = labels;
            this.symbol = symbol;
            this.hint = hint;
        }
        
        public int hashCode() {
            return labels.hashCode() ^ symbol.hashCode() ^ hint.hashCode();
        }
        public boolean equals(Object rhs) {
            if (rhs == null)
                return false;
            if (rhs instanceof Link)
                equals((Link) rhs);
            return false;
        }
        public boolean equals(Link rhs) {
            return labels.equals(rhs.labels) && symbol.equals(rhs.symbol) && hint.equals(rhs.hint);
        }
        
        public String toString() {
            StringBuffer result = new StringBuffer();
            Iterator it = labels.iterator();
            while (it.hasNext()) {
                String label = (String) it.next();
                result.append(label);
                result.append(':');
            }
            result.append(symbol);
            return result.toString();
        }
    }
    
    private static class NondeterministicState {
        public Token tag;
        
        public MultiMap linkToStates = new MultiMap();
        public void link(Link link, NondeterministicState state) {
            linkToStates.add(link, state);
        }
        
        public Set epsilonMoveStates = new LinkedHashSet();
        public void epsilonLink(NondeterministicState state) {
            this.epsilonMoveStates.add(state);
        }
    }
    
    private void link(NondeterministicState initialState, Expression expression, NondeterministicState finalState) {
        link(false, TreeStackSet.EMPTY_SET, TreeStackSet.EMPTY_SET, TreeStackSet.EMPTY_SET, initialState, expression, finalState);
    }
    private void link(boolean embed, TreeStackSet openedAlias, TreeStackSet argumentLabels, TreeStackSet labels, NondeterministicState initialState, Expression expression, NondeterministicState finalState) {
        if (expression instanceof SelectiveExpression) {
            SelectiveExpression exp = (SelectiveExpression) expression;
            Iterator it = exp.operands().iterator();
            while (it.hasNext()) {
                Expression operand = (Expression) it.next();
                link(embed, openedAlias, argumentLabels, labels, initialState, operand, finalState);
            }
        } else if (expression instanceof SequentialExpression) {
            SequentialExpression exp = (SequentialExpression) expression;
            if (exp.operands().isEmpty()) {
                initialState.epsilonLink(finalState);
            } else {
                Iterator it = exp.operands().iterator();
                while (it.hasNext()) {
                    Expression operand = (Expression) it.next();
                    NondeterministicState tmp;
                    if (it.hasNext())
                        tmp = new NondeterministicState();
                    else
                        tmp = finalState;
                    link(embed, openedAlias, argumentLabels, labels, initialState, operand, tmp);
                    initialState = tmp;
                }
            }
        } else if (expression instanceof PlusExpression) {
            PlusExpression exp = (PlusExpression) expression;
            NondeterministicState tmp1 = new NondeterministicState();
            NondeterministicState tmp2 = new NondeterministicState();
            initialState.epsilonLink(tmp1);
            link(embed, openedAlias, argumentLabels, labels, tmp1, exp.operand(), tmp2);
            tmp2.epsilonLink(finalState);
            tmp2.epsilonLink(tmp1);
        } else if (expression instanceof RestrictorExpression) {
            RestrictorExpression exp = (RestrictorExpression) expression;
            link(embed, openedAlias, argumentLabels, labels, initialState, exp.operand(), finalState);
        } else if (expression instanceof LabeledExpression) {
            LabeledExpression exp = (LabeledExpression) expression;
            if (exp.label().getImage().equals("$label"))
                labels = TreeStackSet.pushAll(labels, argumentLabels);
            else
                labels = TreeStackSet.push(labels, exp.label().getImage());
            link(embed, openedAlias, argumentLabels, labels, initialState, exp.operand(), finalState);
        } else if (expression instanceof EmbedExpression) {
            EmbedExpression exp = (EmbedExpression) expression;
            link(true, openedAlias, argumentLabels, labels, initialState, exp.operand(), finalState);
        } else if (expression instanceof TagExpression) {
            TagExpression exp = (TagExpression) expression;
            
            NondeterministicState tmp = new NondeterministicState();
            tmp.tag = exp.tag();
            
            Expression operand = exp.operand();
            if (operand instanceof StringExpression) {
                link(false, openedAlias, argumentLabels, labels, initialState, operand, tmp);
            } else {
                IdentifierExpression x = (IdentifierExpression) operand;
                Definition def = root.getDefinition(x.name());
                if (!(def instanceof TokenReservation)) {
                    logger.error("{0} can''t tag the nonterminal {1}.", exp.tag(), def.identifier());
                }
                link(false, openedAlias, argumentLabels, labels, initialState, x, tmp);
            }
            tmp.epsilonLink(finalState);
            // DEPENDED: TAG_AFTER_NON_EPSILON_MOVE
        } else if (expression instanceof StringExpression) {
            StringExpression exp = (StringExpression) expression;
            Terminal terminal = dfa.getTerminal(exp);
            Link link = new Link(labels, terminal, exp.string().token());
            initialState.link(link, finalState);
        } else {
            IdentifierExpression exp = (IdentifierExpression) expression;
            link(embed, openedAlias, argumentLabels, labels, initialState, exp, finalState);
        }
    }
    private void link(boolean embed, TreeStackSet openedAlias, TreeStackSet argumentLabels, TreeStackSet labels, NondeterministicState initialState, IdentifierExpression exp, NondeterministicState finalState) {
        Definition def = root.getDefinition(exp.name());
        Token hint = exp.name().identifier();
        if (def instanceof TypeDefinition) {
            TypeDefinition d = (TypeDefinition) def;
            if (d.abstractKeyword() != null)
                logger.error("The type {0} is abstract.", exp.name().identifier());
            Link link = new Link(labels, getNonterminal(d), hint);
            initialState.link(link, finalState);
        } else if (def instanceof AliasDefinition) {
            AliasDefinition d = (AliasDefinition) def;
            if (!embed || openedAlias.contains(d)) {
                Link link = new Link(labels, getNonterminal(d), hint);
                initialState.link(link, finalState);
            } else {
                openedAlias = TreeStackSet.push(openedAlias, d);
                if (root.hasExplicitParameterLabel(d)) {
                    argumentLabels = labels;
                    labels = TreeStackSet.EMPTY_SET;
                } else {
                    argumentLabels = TreeStackSet.EMPTY_SET;
                }
                link(false, openedAlias, argumentLabels, labels, initialState, d.expression(), finalState);
            }
        } else {
            TokenReservation d = (TokenReservation) def;
            Link link = new Link(labels, new NamedTerminal(root.getCanonicalName(d)), hint);
            initialState.link(link, finalState);
        }
    }
    
    private static class DeterministicState {
        public Token tag;
        
        public boolean isFinal = false;
        public Map linkToState = new LinkedHashMap();
        
        private static int numberOfInstance = 0;
        public final int idNumber = numberOfInstance++;
        public String toString() {
            return "State" + idNumber;
        }
    }
    
    private Set allStates(Map symbolToInitialState) {
        Set result = new LinkedHashSet();
        allStates(result, symbolToInitialState.values());
        return result;
    }
    private void allStates(Set result, Collection states) {
        Iterator it4 = states.iterator();
        while (it4.hasNext()) {
            DeterministicState state = (DeterministicState) it4.next();
            if (result.add(state))
                allStates(result, state.linkToState.values());
        }
    }
    
    private void reportStates(Map symbolToInitialState, Set allStates) {
        if (logger.verboseLevel() < Logger.LEVEL_VERBOSEST)
            return;
        
        logger.verbosest(environment.processingFile, "----States Report----");
        Iterator it = symbolToInitialState.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = (Map.Entry) it.next();
            Symbol symbol = (Symbol) entry.getKey();
            DeterministicState state = (DeterministicState) entry.getValue();
            logger.verbosest("{0} => x{1}", symbol, state);
        }
        it = allStates.iterator();
        while (it.hasNext()) {
            DeterministicState state = (DeterministicState) it.next();
            logger.verbosest("-- {0} --", state);
            
            if (state.isFinal)
                logger.verbosest("    FINAL STATE");
            if (state.tag != null)
                logger.verbosest("    TAG: " + state.tag);
            Iterator it2 = state.linkToState.entrySet().iterator();
            while (it2.hasNext()) {
                Map.Entry entry = (Map.Entry) it2.next();
                Link link = (Link) entry.getKey();
                DeterministicState s = (DeterministicState) entry.getValue();
                logger.verbosest("        {0} ==> x{1}", link, s);
            }
        }
    }
    
    
    private class DeterministicConverter {
        private Map substatesToState = new LinkedHashMap();
        private DeterministicState createDeterministicState(Set substates, NondeterministicState finalState) {
            DeterministicState result = (DeterministicState) substatesToState.get(substates);
            if (result == null) {
                result = new DeterministicState();
                substatesToState.put(substates, result);
                bind(substates, finalState, result);
            }
            return result;
        }
        
        private void bind(Set substates, NondeterministicState finalState, DeterministicState target) {
            Set closure = getEpsilonClosure(substates);
            
            // set tag
            Set tags = new LinkedHashSet();
            Iterator it = substates.iterator();
            while (it.hasNext()) {
                NondeterministicState s = (NondeterministicState) it.next();
                tags.add(s.tag);    // DEPENDS: TAG_AFTER_NON_EPSILON_MOVE
            }
            if (!tags.isEmpty()) {
                if (tags.size() == 1) {
                    target.tag = (Token) tags.iterator().next();
                } else {
                    boolean first = true;
                    it = tags.iterator();
                    while (it.hasNext()) {
                        Token tag = (Token) it.next();
                        if (tag != null) {
                            if (first) {
                                logger.error(tag, "An unified state has different tag(s):");
                                first = false;
                            }
                            logger.error("        {0}", tag);
                        }
                    }
                }
            }
            
            // set final
            target.isFinal = closure.contains(finalState);
            
            // set map
            MultiMap pairToLinks = new MultiMap();
            MultiMap pairToSubstates = new MultiMap();
            Iterator it29 = closure.iterator();
            while (it29.hasNext()) {
                NondeterministicState substate = (NondeterministicState) it29.next();
                Iterator it3 = substate.linkToStates.entrySet().iterator();
                while (it3.hasNext()) {
                    Map.Entry entry = (Map.Entry) it3.next();
                    Link link = (Link) entry.getKey();
                    Set states = (Set) entry.getValue();
                    List pair = Arrays.asList(new Object[] { link.labels, link.symbol });
                    pairToLinks.add(pair, link);
                    pairToSubstates.addAll(pair, states);
                }
            }
            
            MultiMap linkToSubstates = new MultiMap();
            Iterator it30 = pairToLinks.entrySet().iterator();
            while (it30.hasNext()) {
                Map.Entry entry = (Map.Entry) it30.next();
                List pair = (List) entry.getKey();
                Set links = (Set) entry.getValue();
                Set states = pairToSubstates.getValueSet(pair);
                
                DeterministicState newState = createDeterministicState(states, finalState);
                target.linkToState.put((Link) links.iterator().next(), newState);
            }
        }
        
        private Set getEpsilonClosure(Set states) {
            Set result = new LinkedHashSet();
            getEpsilonClosure(result, states);
            return result;
        }
        private void getEpsilonClosure(Set result, Set states) {
            Iterator it1 = states.iterator();
            while (it1.hasNext()) {
                NondeterministicState state = (NondeterministicState) it1.next();
                if (result.add(state))
                    getEpsilonClosure(result, state.epsilonMoveStates);
            }
        }
    }
    
    private DeterministicState buildMinimalForm(DeterministicState initialState) {
        Map symbolToInitialState = Collections.singletonMap(null, initialState);
        Set allStates = allStates(symbolToInitialState);
        symbolToInitialState = buildMinimalForm(symbolToInitialState, allStates);
        return (DeterministicState) symbolToInitialState.get(null);
    }
    private Map buildMinimalForm(Map symbolToInitialState, Set allStates) {
        Set initialStates = new HashSet(symbolToInitialState.values());
        
        // initialize
        List groups = new ArrayList(allStates.size());
        Map stateToGroupID = new HashMap();
        {
            Map tagAndFinalToGroupID = new LinkedHashMap();
            Iterator it = allStates.iterator();
            while (it.hasNext()) {
                DeterministicState state = (DeterministicState) it.next();
                String tag = (state.tag == null) ? null : state.tag.getImage();
                Object key = Arrays.asList(new Object[] { tag, new Boolean(state.isFinal) });
                Integer groupID;
                if (initialStates.contains(state)) {
                    groupID = new Integer(groups.size());
                    groups.add(new LinkedList());
                } else {
                    groupID = (Integer) tagAndFinalToGroupID.get(key);
                    if (groupID == null) {
                        groupID = new Integer(groups.size());
                        groups.add(new LinkedList());
                        tagAndFinalToGroupID.put(key, groupID);
                    }
                }
                List states = (List) groups.get(groupID.intValue());
                states.add(state);
                stateToGroupID.put(state, groupID);
            }
        }
        
        // loop
        int lastMoodifiedIndex = -1;
        int index = 0;
        for (;; index++) {
            if (index >= groups.size()) {
                if (lastMoodifiedIndex == -1)
                    break;
                index -= groups.size();
            }
            
            List group = (List) groups.get(index);
            assert !group.isEmpty();
            if (group.size() <= 1) {
                if (index == lastMoodifiedIndex)
                    break;
                continue;
            }
            
            Map keyToStateList = new LinkedHashMap();
            Iterator it = group.iterator();
            while (it.hasNext()) {
                DeterministicState state = (DeterministicState) it.next();
                Set key = new HashSet();
                Iterator it2 = state.linkToState.entrySet().iterator();
                while (it2.hasNext()) {
                    Map.Entry entry = (Map.Entry) it2.next();
                    Link link = (Link) entry.getKey();
                    DeterministicState nextState = (DeterministicState) entry.getValue();
                    key.add(Arrays.asList(new Object[] { link.symbol, link.labels, stateToGroupID.get(nextState) }));
                }
                List list = (List) keyToStateList.get(key);
                if (list == null) {
                    list = new LinkedList();
                    keyToStateList.put(key, list);
                }
                list.add(state);
            }
            
            if (keyToStateList.size() <= 1) {
                if (index == lastMoodifiedIndex)
                    break;
                continue;
            }
            
            lastMoodifiedIndex = index;
            it = keyToStateList.values().iterator();
            while (it.hasNext()) {
                List states = (List) it.next();
                if (it.hasNext()) {
                    Integer newID = new Integer(groups.size());
                    groups.add(states);
                    Iterator it2 = states.iterator();
                    while (it2.hasNext()) {
                        DeterministicState state = (DeterministicState) it2.next();
                        stateToGroupID.put(state, newID);
                    }
                } else {
                    groups.set(index, states);
                }
            }
        }
        
        // create result states.
        DeterministicState[] result = new DeterministicState[groups.size()];
        for (int i = 0; i < result.length; i++)
            result[i] = new DeterministicState();
        
        for (int i = 0; i < result.length; i++) {
            List group = (List) groups.get(i);
            DeterministicState hot = result[i];
            
            /*logger.verbosest(environment.processingFile, "x{0} <=", hot);
            Iterator it32 = group.iterator();
            while (it32.hasNext()) {
                DeterministicState old = (DeterministicState) it32.next();
                logger.verbosest(environment.processingFile, "        x{0}", old);
            }*/
            
            DeterministicState old = (DeterministicState) group.get(0);
            hot.tag = old.tag;
            
            hot.isFinal = old.isFinal;
            Iterator it46 = old.linkToState.entrySet().iterator();
            while (it46.hasNext()) {
                Map.Entry entry = (Map.Entry) it46.next();
                Link link = (Link) entry.getKey();
                DeterministicState nextState = (DeterministicState) entry.getValue();
                Integer nextGroupID = (Integer) stateToGroupID.get(nextState);
                
                DeterministicState hotNext = result[nextGroupID.intValue()];
                hot.linkToState.put(link, hotNext);
            }
        }
        
        // create result map.
        Map resultMap = new LinkedHashMap();
        Iterator it = symbolToInitialState.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = (Map.Entry) it.next();
            Symbol symbol = (Symbol) entry.getKey();
            DeterministicState state = (DeterministicState) entry.getValue();
            Integer groupID = (Integer) stateToGroupID.get(state);
            resultMap.put(symbol, result[groupID.intValue()]);
        }
        return resultMap;
    }
    
    private Map stateToMapPreviousStateToLinks(Set allStates) {
        Map result = new LinkedHashMap();
        Iterator it13 = allStates.iterator();
        while (it13.hasNext()) {
            DeterministicState state = (DeterministicState) it13.next();
            Iterator it47 = state.linkToState.entrySet().iterator();
            while (it47.hasNext()) {
                Map.Entry entry = (Map.Entry) it47.next();
                Link link = (Link) entry.getKey();
                DeterministicState nextState = (DeterministicState) entry.getValue();
                
                MultiMap map = (MultiMap) result.get(nextState);
                if (map == null) {
                    map = new MultiMap();
                    result.put(nextState, map);
                }
                map.add(state, link);
            }
        }
        return result;
    }
    
    private static final Set LABEL_INHERITOR = Collections.singleton("$label");
    private Map buildCFG(Map symbolToInitialState, Set allStates) {
        Map result = new LinkedHashMap(symbolToInitialState);
        
        Map stateToMapPreviousStateToLinks = stateToMapPreviousStateToLinks(allStates);
        
        Iterator it17 = stateToMapPreviousStateToLinks.entrySet().iterator();
        while (it17.hasNext()) {
            Map.Entry entry = (Map.Entry) it17.next();
            DeterministicState state = (DeterministicState) entry.getKey();
            MultiMap map = (MultiMap) entry.getValue();
            
            int count = 0;
            Iterator it18 = map.entrySet().iterator();
            while (it18.hasNext()) {
                Map.Entry entry2 = (Map.Entry) it18.next();
                Set links = (Set) entry2.getValue();
                count += links.size();
            }
            if (count > 1) {
                AnonymousNonterminal anony = new AnonymousNonterminal();
                boolean used = false;
                
                Iterator it19 = map.entrySet().iterator();
                while (it19.hasNext()) {
                    Map.Entry entry2 = (Map.Entry) it19.next();
                    DeterministicState prevState = (DeterministicState) entry2.getKey();
                    Set links = (Set) entry2.getValue();
                    Iterator it20 = links.iterator();
                    while (it20.hasNext()) {
                        Link link = (Link) it20.next();
                        
                        DeterministicState newState = new DeterministicState();
                        prevState.linkToState.put(link, newState);
                        newState.tag = state.tag;
                        if (state.isFinal)
                            newState.isFinal = true;
                        
                        if (!state.linkToState.isEmpty()) {
                            DeterministicState finalState = new DeterministicState();
                            newState.linkToState.put(new Link(LABEL_INHERITOR, anony, link.hint), finalState);
                            finalState.isFinal = true;
                            used = true;
                        }
                    }
                }
                
                state.tag = null;
                state.isFinal = false;
                if (used)
                    result.put(anony, state);
            }
        }
        
        return result;
    }
    
    private static interface SimpleState {
        public Map symbolToNextState();
    }
    
    private static class LRState implements LRTable.LRState, SimpleState {
        public Map symbolToNextState = new LinkedHashMap();
        public Map symbolToNextState() {
            return symbolToNextState;
        }
        public MultiMap symbolToReductions = null;
        public Map symbolToReductions() {
            return symbolToReductions;
        }
        
        public Token tag;
        public Token tag() {
            return tag;
        }
        
        public Set kernelItems = new LinkedHashSet();
        public Set items = new LinkedHashSet();
        public Map coreToItem = new LinkedHashMap();
        public boolean containsFinal = false;
        
        private static int numberOfInstance = 0;
        public final int idNumber = numberOfInstance++;
        public String toString() {
            return "LRState" + idNumber;
        }
    }
    private static class Item {
        public final LRState owner;
        public final DeterministicState core;
        public final boolean kernel;
        
        public Set lookaheads = null;
        public MultiMap symbolToNextItems = new MultiMap();
        public Set deliveredItems = new LinkedHashSet();
        
        public Item(LRState owner, DeterministicState core, boolean kernel) {
            assert owner != null && core != null;
            this.owner = owner;
            this.core = core;
            this.kernel = kernel;
        }
        
        public String toString() {
            String k = kernel ? " (kernel)" : "";
            String f = core.isFinal ? " (FINAL)" : "";
            return core + f + " ; " + lookaheads + k;
        }
    }
    
    public static class Reduction implements LRTable.Reduction {
        private Nonterminal reducedSymbol;
        private List labelSetList;
        public Reduction(Nonterminal reducedSymbol, List labelSetList) {
            this.reducedSymbol = reducedSymbol;
            this.labelSetList = labelSetList;
        }
        public Nonterminal reducedSymbol() {
            return reducedSymbol;
        }
        public List labelSetList() {
            return labelSetList;
        }
        
        public int hashCode() {
            return reducedSymbol.hashCode() + labelSetList.hashCode();
        }
        public final boolean equals(Object rhs) {
            if (rhs instanceof Reduction)
                return equals((Reduction) rhs);
            return false;
        }
        public boolean equals(Reduction rhs) {
            return reducedSymbol.equals(rhs.reducedSymbol) && labelSetList.equals(rhs.labelSetList);
        }
        
        public String toString() {
            return reducedSymbol + " : " + labelSetList;
        }
    }
    
    private class LRHelper {
        private Map symbolToInitialLRState;
        private LRState finalState;
        public LRHelper(Map symbolToInitialLRState, LRState finalState) {
            this.symbolToInitialLRState = symbolToInitialLRState;
            this.finalState = finalState;
        }
        
        public Map symbolToInitialLRState() {
            return symbolToInitialLRState;
        }
        public LRState finalState() {
            return finalState;
        }
        
        private MultiMap backsteps = null;
        public Set backstep(LRState state) {
            allLRStates();
            Set result = backsteps.getValueSet(state);
            if (result == null) {
                result = Collections.EMPTY_SET;
                backsteps.put(state, result);
            }
            return result;
        }
        
        private Set allLRStates = null;
        public Set allLRStates() {
            if (allLRStates == null) {
                backsteps = new MultiMap();
                allLRStates = new LinkedHashSet();
                allLRStates.add(finalState);
                LinkedList undone = new LinkedList(symbolToInitialLRState.values());
                while (!undone.isEmpty()) {
                    LRState state = (LRState) undone.removeFirst();
                    if (allLRStates.add(state)) {
                        Iterator it59 = state.symbolToNextState.values().iterator();
                        while (it59.hasNext()) {
                            LRState nextState = (LRState) it59.next();
                            backsteps.add(nextState, state);
                        }
                        undone.addAll(state.symbolToNextState.values());
                    }
                }
            }
            return allLRStates;
        }
        
        private Map initialLRStateToSymbol = null;
        public Map initialLRStateToSymbol() {
            if (initialLRStateToSymbol == null) {
                initialLRStateToSymbol = new LinkedHashMap();
                Iterator it44 = symbolToInitialLRState.entrySet().iterator();
                while (it44.hasNext()) {
                    Map.Entry entry = (Map.Entry) it44.next();
                    initialLRStateToSymbol.put(entry.getValue(), entry.getKey());
                }
                assert initialLRStateToSymbol.size() == symbolToInitialLRState.size();
            }
            return initialLRStateToSymbol;
        }
        
        private MultiMap itemToGeneratorKernels = null;
        public MultiMap itemToGeneratorKernels() {
            if (itemToGeneratorKernels == null) {
                itemToGeneratorKernels = new MultiMap();
                Iterator it12 = allLRStates().iterator();
                while (it12.hasNext()) {
                    LRState state = (LRState) it12.next();
                    Iterator it31 = state.kernelItems.iterator();
                    while (it31.hasNext()) {
                        Item kernel = (Item) it31.next();
                        Set done = new HashSet();
                        LinkedList undone = new LinkedList();
                        undone.add(kernel);
                        while (!undone.isEmpty()) {
                            Item item = (Item) undone.removeFirst();
                            if (done.add(item)) {
                                Iterator it33 = item.deliveredItems.iterator();
                                while (it33.hasNext()) {
                                    Item deliveredItem = (Item) it33.next();
                                    itemToGeneratorKernels.add(deliveredItem, kernel);
                                    undone.add(deliveredItem);
                                }
                            }
                        }
                    }
                }
            }
            return itemToGeneratorKernels;
        }
        public Set getGeneratorKernels(Item item) {
            return itemToGeneratorKernels().getValueSet(item);
        }
        
        private Map stateToAscendedStateAndSymbol = null;
        public Map stateToAscendedStateAndSymbol() {
            if (stateToAscendedStateAndSymbol == null) {
                stateToAscendedStateAndSymbol = new LinkedHashMap();
                
                LinkedList undone = new LinkedList();
                Iterator it39 = symbolToInitialLRState.values().iterator();
                while (it39.hasNext()) {
                    LRState state = (LRState) it39.next();
                    stateToAscendedStateAndSymbol.put(state, null);
                    undone.add(state);
                }
                
                while (!undone.isEmpty()) {
                    LRState state = (LRState) undone.removeFirst();
                    Iterator it38 = state.symbolToNextState.entrySet().iterator();
                    while (it38.hasNext()) {
                        Map.Entry entry = (Map.Entry) it38.next();
                        Symbol symbol = (Symbol) entry.getKey();
                        LRState nextState = (LRState) entry.getValue();
                        if (!stateToAscendedStateAndSymbol.containsKey(nextState)) {
                            stateToAscendedStateAndSymbol.put(nextState, new Object[] { state, symbol });
                            undone.add(nextState);
                        }
                    }
                }
            }
            return stateToAscendedStateAndSymbol;
        }
        
    }
    
    /*private Set allLRStates(Map symbolToInitialLRState) {
        Set result = new LinkedHashSet();
        allLRStates(result, symbolToInitialLRState.values());
        return result;
    }
    private void allLRStates(Set result, Collection states) {
        Iterator it5 = states.iterator();
        while (it5.hasNext()) {
            LRState state = (LRState) it5.next();
            if (result.add(state))
                allLRStates(result, state.symbolToNextState.values());
        }
    }*/
    
    private void reportLRStates(LRHelper lrHelper) {
        if (logger.verboseLevel() < Logger.LEVEL_VERBOSEST)
            return;
        
        logger.verbosest(environment.processingFile, "----LRStates Report----");
        
        Iterator it = lrHelper.symbolToInitialLRState().entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = (Map.Entry) it.next();
            Symbol symbol = (Symbol) entry.getKey();
            LRState state = (LRState) entry.getValue();
            logger.verbosest("{0} => x{1}", symbol, state);
        }
        
        it = lrHelper.allLRStates().iterator();
        while (it.hasNext()) {
            LRState state = (LRState) it.next();
            logger.verbosest("-- {0} --", state);
            if (state.tag != null)
                logger.verbosest("        TAG: " + state.tag);
            Iterator it2 = state.symbolToNextState.entrySet().iterator();
            while (it2.hasNext()) {
                Map.Entry entry = (Map.Entry) it2.next();
                Symbol symbol = (Symbol) entry.getKey();
                LRState s = (LRState) entry.getValue();
                logger.verbosest("        {0} ==> x{1}", symbol, s);
            }
            logger.verbosest("    -- items --");
            Iterator it34 = state.items.iterator();
            while (it34.hasNext()) {
                Item item = (Item) it34.next();
                logger.verbosest("        x{0}", item);
                
                if (item.core.isFinal)
                    logger.verbosest("            FINAL STATE");
                Iterator it35 = item.core.linkToState.entrySet().iterator();
                while (it35.hasNext()) {
                    Map.Entry entry = (Map.Entry) it35.next();
                    Link link = (Link) entry.getKey();
                    DeterministicState s = (DeterministicState) entry.getValue();
                    logger.verbosest("            {0} ==> x{1}", link, s);
                }
                
            }
            if (state.symbolToReductions != null) {
                logger.verbosest("    -- reductions --");
                Iterator it14 = state.symbolToReductions.entrySet().iterator();
                while (it14.hasNext()) {
                    Map.Entry entry = (Map.Entry) it14.next();
                    Symbol symbol = (Symbol) entry.getKey();
                    Set reductions = (Set) entry.getValue();
                    logger.verbosest("        {0} ==>", symbol);
                    Iterator it15 = reductions.iterator();
                    while (it15.hasNext()) {
                        Reduction reduction = (Reduction) it15.next();
                        logger.verbosest("            {0}", reduction);
                    }
                }
            }
        }
    }
    
    private class LRStateBuilder {
        private Map symbolToInitialState;
        public LRStateBuilder(Map symbolToInitialState) {
            this.symbolToInitialState = symbolToInitialState;
        }
        
        private Map kernelCoresToLRState = new HashMap();
        private LRState createLRState(Set kernelCores) {
            LRState result = (LRState) kernelCoresToLRState.get(kernelCores);
            if (result != null)
                return result;  // result.coreToItem shuld be set.
            
            // create and register.
            result = new LRState();
            kernelCoresToLRState.put(kernelCores, result);
            
            {
                Iterator it = kernelCores.iterator();
                while (it.hasNext()) {
                    DeterministicState kernelCore = (DeterministicState) it.next();
                    Item kernel = new Item(result, kernelCore, true);
                    result.kernelItems.add(kernel);
                    result.coreToItem.put(kernelCore, kernel);
                }
            }
            
            // list the next kernelCores.
            MultiMap symbolToNextKernelCores = new MultiMap();
            LinkedList undone = new LinkedList(result.kernelItems);
            while (!undone.isEmpty()) {
                Item item = (Item) undone.removeFirst();
                result.items.add(item);
                if (item.core.isFinal)
                    result.containsFinal = true;
                Iterator it = item.core.linkToState.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry entry = (Map.Entry) it.next();
                    Link link = (Link) entry.getKey();
                    DeterministicState nextKernelCore = (DeterministicState) entry.getValue();
                    Symbol symbol = link.symbol;
                    symbolToNextKernelCores.add(symbol, nextKernelCore);
                    if (symbol instanceof Nonterminal) {
                        DeterministicState core = (DeterministicState) symbolToInitialState.get(symbol);
                        Item deliveredItem = (Item) result.coreToItem.get(core);
                        if (deliveredItem == null) {
                            deliveredItem = new Item(result, core, false);
                            result.coreToItem.put(core, deliveredItem);
                            undone.add(deliveredItem);
                        }
                        item.deliveredItems.add(deliveredItem);
                    }
                }
            }
            // create next states.
            Iterator it = symbolToNextKernelCores.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                Symbol symbol = (Symbol) entry.getKey();
                Set nextKernelCores = (Set) entry.getValue();
                LRState nextState = createLRState(nextKernelCores);
                result.symbolToNextState.put(symbol, nextState);
            }
            // link items.
            Iterator it16 = result.items.iterator();
            while (it16.hasNext()) {
                Item item = (Item) it16.next();
                Iterator it48 = item.core.linkToState.entrySet().iterator();
                while (it48.hasNext()) {
                    Map.Entry entry = (Map.Entry) it48.next();
                    Link link = (Link) entry.getKey();
                    DeterministicState nextCore = (DeterministicState) entry.getValue();
                    
                    LRState nextState = (LRState) result.symbolToNextState.get(link.symbol);
                    Item nextItem = (Item) nextState.coreToItem.get(nextCore);
                    item.symbolToNextItems.add(link.symbol, nextItem);
                }
                
            }
            
            return result;
        }
    }
    
    private void tagLRStates(LRHelper lrHelper, Map initialStateToSymbol) {
        Iterator it7 = lrHelper.allLRStates().iterator();
        while (it7.hasNext()) {
            LRState s = (LRState) it7.next();
            Set tags = new LinkedHashSet();
            Set tagImages = new LinkedHashSet();
            Iterator it10 = s.kernelItems.iterator();
            while (it10.hasNext()) {
                Item kernelItem = (Item) it10.next();
                Token tag = kernelItem.core.tag;
                tags.add(tag);
                tagImages.add((tag == null) ? null : tag.getImage());
            }
            s.tag = (Token) tags.iterator().next();
            if (tagImages.size() > 1) {
                Iterator it = getExamplePath(s, lrHelper, initialStateToSymbol).iterator();
                Nonterminal nonterminal = (Nonterminal) it.next();
                it.next();  // state
                boolean hasNoTagging = tags.remove(null);
                StringBuffer buffer = new StringBuffer();
                while (it.hasNext()) {
                    Symbol symbol = (Symbol) it.next();
                    it.next();  // state
                    buffer.append(symbol);
                    if (it.hasNext())
                        buffer.append(' ');
                }
                Token hint = getHint(nonterminal);
                logger.error("ambiguous tag: {0} with the input", hint);
                logger.error(hint, "        {0}", buffer);
                logger.error(hint, "    has the different tagging:");
                if (hasNoTagging)
                    logger.error(hint, "        no tagging");
                Iterator it11 = tags.iterator();
                while (it11.hasNext()) {
                    Token tag = (Token) it11.next();
                    logger.error("        {0}", tag);
                }
            }
        }
    }
    
    private class LookaheadDeliverer {
        private Map symbolToInitialState;
        private Set allStates;
        public LookaheadDeliverer(Map symbolToInitialState, Set allStates) {
            this.symbolToInitialState = symbolToInitialState;
            this.allStates = allStates;
            buildEmptyAcceptableSet();
            buildFirstSet();
        }
        
        private Set emptyAcceptable;
        private void buildEmptyAcceptableSet() {
            emptyAcceptable = new HashSet();
            Set unfixed = new LinkedHashSet(allStates);
            boolean modified = true;
            while (!unfixed.isEmpty() && modified) {
                modified = false;
                Iterator outer = unfixed.iterator();
                while (outer.hasNext()) {
                    DeterministicState state = (DeterministicState) outer.next();
                    // We try to know whether the state accepts the empty input.
                    
                    int check = -1;      // -1=unmatch the empty input, 0=unfixed, 1=match the empty input.
                    if (state.isFinal) {
                        check = 1;
                    } else {
                        Iterator inner = state.linkToState.entrySet().iterator();
                        while (inner.hasNext()) {
                            Map.Entry entry = (Map.Entry) inner.next();
                            Link link = (Link) entry.getKey();
                            Symbol symbol = link.symbol;
                            if (symbol instanceof Nonterminal) {
                                DeterministicState nextState = (DeterministicState) entry.getValue();
                                DeterministicState s = (DeterministicState) symbolToInitialState.get(symbol);
                                if (unfixed.contains(nextState) || unfixed.contains(s)) {
                                    check = 0;
                                } else {
                                    if (emptyAcceptable.contains(nextState) && emptyAcceptable.contains(s)) {
                                        check = 1;
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    if (check != 0) {
                        outer.remove();
                        modified = true;
                        if (check == 1) {
                            emptyAcceptable.add(state);
                        }
                    }
                }
            }
            
            /*if (logger.verboseLevel() >= Logger.LEVEL_VERBOSEST) {
                logger.verbosest(environment.processingFile, "buildEmptyAcceptableSet");
                Iterator it8 = emptyAcceptable.iterator();
                while (it8.hasNext()) {
                    DeterministicState state = (DeterministicState) it8.next();
                    logger.verbosest("        x{0}", state);
                }
            }*/
        }
        public boolean isEmptyAcceptable(DeterministicState state) {
            return emptyAcceptable.contains(state);
        }
        public boolean isEmptyAcceptable(Nonterminal symbol) {
            DeterministicState state = (DeterministicState) symbolToInitialState.get(symbol);
            return isEmptyAcceptable(state);
        }
        public boolean isEmptyAcceptable(Symbol symbol) {
            if (symbol instanceof Terminal)
                return false;
            return isEmptyAcceptable((Nonterminal) symbol);
        }
        
        private MultiMap stateToFirstSet;
        private void buildFirstSet() {
            stateToFirstSet = new MultiMap();
            {
                Iterator it0 = allStates.iterator();
                while (it0.hasNext()) {
                    DeterministicState state = (DeterministicState) it0.next();
                    stateToFirstSet.put(state, new LinkedHashSet());
                }
            }
            
            Set unfixed = new LinkedHashSet(allStates);
            boolean modified = true;
            while (!unfixed.isEmpty() && modified) {
                modified = false;
                Iterator outer = unfixed.iterator();
                while (outer.hasNext()) {
                    DeterministicState state = (DeterministicState) outer.next();
                    Set first = stateToFirstSet.getValueSet(state);
                    boolean fixed = true;
                    Iterator inner = state.linkToState.entrySet().iterator();
                    while (inner.hasNext()) {
                        Map.Entry entry = (Map.Entry) inner.next();
                        Link link = (Link) entry.getKey();
                        if (link.symbol instanceof Terminal) {
                            modified |= first.add(link.symbol);
                        } else {
                            DeterministicState ds = (DeterministicState) symbolToInitialState.get(link.symbol);
                            modified |= first.addAll(stateToFirstSet.getValueSet(ds));
                            if (unfixed.contains(ds))
                                fixed = false;
                            
                            if (isEmptyAcceptable(ds)) {
                                ds = (DeterministicState) entry.getValue();
                                modified |= first.addAll(stateToFirstSet.getValueSet(ds));
                                if (unfixed.contains(ds))
                                    fixed = false;
                            }
                        }
                    }
                    if (fixed) {
                        outer.remove();
                        modified = true;
                    }
                }
            }
            
            /*if (logger.verboseLevel() >= Logger.LEVEL_VERBOSEST) {
                logger.verbosest(environment.processingFile, "buildFirstSet");
                Iterator it8 = stateToFirstSet.entrySet().iterator();
                while (it8.hasNext()) {
                    Map.Entry entry = (Map.Entry) it8.next();
                    DeterministicState state = (DeterministicState) entry.getKey();
                    Set first = (Set) entry.getValue();
                    logger.verbosest("        x{0}  {1}", state, first);
                }
            }*/
        }
        public Set first(DeterministicState state) {
            return stateToFirstSet.getValueSet(state);
        }
        
        public void deliverLookahead(LRHelper lrHelper) {
            Iterator it36 = lrHelper.allLRStates().iterator();
            while (it36.hasNext()) {
                LRState state = (LRState) it36.next();
                Iterator it37 = state.items.iterator();
                while (it37.hasNext()) {
                    Item item = (Item) it37.next();
                    assert item.lookaheads == null;
                    item.lookaheads = new LinkedHashSet();
                }
            }
            
            LinkedList jobs = new LinkedList();
            MultiMap itemToLookaheadDependencies = new MultiMap();
            {
                LinkedList undone = new LinkedList();
                {
                    Iterator it4 = lrHelper.symbolToInitialLRState().values().iterator();
                    while (it4.hasNext()) {
                        LRState state = (LRState) it4.next();
                        undone.addAll(state.kernelItems);
                    }
                }
                Set done = new HashSet();
                while (!undone.isEmpty()) {
                    Item item = (Item) undone.removeFirst();
                    if (done.add(item)) {
                        Iterator it3 = item.symbolToNextItems.entrySet().iterator();
                        while (it3.hasNext()) {
                            Map.Entry entry = (Map.Entry) it3.next();
                            Symbol symbol = (Symbol) entry.getKey();
                            Set nextItems = (Set) entry.getValue();
                            Iterator it50 = nextItems.iterator();
                            while (it50.hasNext()) {
                                Item nextItem = (Item) it50.next();
                                itemToLookaheadDependencies.add(item, nextItem);
                                undone.add(nextItem);
                                
                                if (symbol instanceof Nonterminal) {
                                    DeterministicState core = (DeterministicState) symbolToInitialState.get(symbol);
                                    Item deliveredItem = (Item) item.owner.coreToItem.get(core);
                                    jobs.add(new Object[] { deliveredItem, first(nextItem.core) });
                                    undone.add(deliveredItem);
                                    if (isEmptyAcceptable(nextItem.core))
                                        itemToLookaheadDependencies.add(item, deliveredItem);
                                }
                            }
                        }
                    }
                }
            }
            
            /*Iterator it9 = itemToLookaheadDependencies.entrySet().iterator();
            while (it9.hasNext()) {
                Map.Entry entry = (Map.Entry) it9.next();
                List key = (List) entry.getKey();
                Set values = (Set) entry.getValue();
                System.out.println(key + " -> " + values);
            }*/
            
            Iterator it1 = jobs.iterator();
            while (it1.hasNext()) {
                Object[] job = (Object[]) it1.next();
                Item item = (Item) job[0];
                Set lookaheads = (Set) job[1];
                deliverLookahead(itemToLookaheadDependencies, item, lookaheads);
            }
        }
        private void deliverLookahead(MultiMap itemToLookaheadDependencies, Item item, Set lookaheads) {
            if (item.lookaheads.addAll(lookaheads)) {
                Set dependencies = itemToLookaheadDependencies.getValueSet(item);
                if (dependencies != null) {
                    Iterator it6 = dependencies.iterator();
                    while (it6.hasNext()) {
                        Item dependency = (Item) it6.next();
                        deliverLookahead(itemToLookaheadDependencies, dependency, lookaheads);
                    }
                }
            }
        }
    }
    
    private class ReductionGenerator {
        private Map initialStateToSymbol;
        private Map stateToMapPreviousStateToLinks;
        public ReductionGenerator(Map symbolToInitialState, Map initialStateToSymbol, Set allStates) {
            this.initialStateToSymbol = initialStateToSymbol;
            stateToMapPreviousStateToLinks = stateToMapPreviousStateToLinks(allStates);
        }
        
        public Set generateReductions(LRHelper lrHelper) {
            Set conflictedStates = new LinkedHashSet();
            Iterator it21 = lrHelper.allLRStates().iterator();
            while (it21.hasNext()) {
                LRState state = (LRState) it21.next();
                assert state.symbolToReductions == null;
                state.symbolToReductions = new MultiMap();
                boolean conflicted = false;
                if (state.containsFinal) {
                    Iterator it5 = state.items.iterator();
                    while (it5.hasNext()) {
                        Item item = (Item) it5.next();
                        if (item.core.isFinal) {
                            Reduction reduction = getReduction(item);
                            Iterator it24 = item.lookaheads.iterator();
                            while (it24.hasNext()) {
                                Symbol lookahead = (Symbol) it24.next();
                                state.symbolToReductions.add(lookahead, reduction);
                            }
                        }
                    }
                    if (!conflicted) {
                        Set shiftableSymbol = state.symbolToNextState.keySet();
                        Iterator it26 = state.symbolToReductions.entrySet().iterator();
                        while (it26.hasNext()) {
                            Map.Entry entry = (Map.Entry) it26.next();
                            Symbol symbol = (Symbol) entry.getKey();
                            Set reductions = (Set) entry.getValue();
                            if (shiftableSymbol.contains(symbol) || reductions.size() > 1)
                                conflicted = true;
                        }
                    }
                }
                if (conflicted)
                    conflictedStates.add(state);
            }
            
            return conflictedStates;
        }
        
        private Reduction getReduction(Item item) {
            return getReduction(item.core);
        }
        private Map stateToReduction = new HashMap();
        private Reduction getReduction(DeterministicState finalState) {
            DeterministicState state = finalState;
            Reduction result = (Reduction) stateToReduction.get(finalState);
            if (result == null) {
                LinkedList labelsList = new LinkedList();
                
                for (;;) {
                    Nonterminal symbol = (Nonterminal) initialStateToSymbol.get(state);
                    if (symbol != null) {
                        result = new Reduction(symbol, labelsList);
                        break;
                    }
                    
                    MultiMap map = (MultiMap) stateToMapPreviousStateToLinks.get(state);
                    assert map.size() == 1;
                    state = (DeterministicState) map.keySet().iterator().next();
                    Set links = (Set) map.values().iterator().next();
                    assert links.size() == 1;
                    Link link = (Link) links.iterator().next();
                    
                    labelsList.addFirst(link.labels);
                }
                
                stateToReduction.put(finalState, result);
            }
            return result;
        }
    }
    
    private class ConflictChaser {
        private LRHelper lrHelper;
        private ReductionGenerator generator;
        private Map initialStateToSymbol;
        public ConflictChaser(LRHelper lrHelper, ReductionGenerator generator, Map initialStateToSymbol) {
            this.lrHelper = lrHelper;
            this.generator = generator;
            this.initialStateToSymbol = initialStateToSymbol;
        }
        public int chaseConflicts(Set conflictedStates) {
            int result = 0;
            Iterator it67 = conflictedStates.iterator();
            while (it67.hasNext()) {
                LRState conflictedState = (LRState) it67.next();
                if (chaseConflict(conflictedState))
                    result++;
            }
            return result;
        }
        private boolean chaseConflict(LRState conflictedState) {
            boolean result = false;
            Set nextSymbols = new LinkedHashSet();
            {
                Set shiftableSymbols = conflictedState.symbolToNextState.keySet();
                Iterator it26 = conflictedState.symbolToReductions.entrySet().iterator();
                while (it26.hasNext()) {
                    Map.Entry entry = (Map.Entry) it26.next();
                    Symbol symbol = (Symbol) entry.getKey();
                    Set reductions = (Set) entry.getValue();
                    if (shiftableSymbols.contains(symbol) || reductions.size() > 1)
                        nextSymbols.add(symbol);
                }
            }
            
            Set safeLookaheadPaths = new LinkedHashSet();
            Set lookaheadPaths = new LinkedHashSet();
            Iterator it = nextSymbols.iterator();
            while (it.hasNext()) {
                Symbol lookahead = (Symbol) it.next();
                /**/
                List maybeConfictPath = getMaybeConflictPath(conflictedState, lookahead);
                if (maybeConfictPath == null) {
                    it.remove();
                    safeLookaheadPaths.add(Collections.singletonList(lookahead));
                } else {
                    result = true;
                    lookaheadPaths.add(maybeConfictPath);
                }
                /**
                if (isResolvableConflict(conflictedState, lookahead)) {
                    it.remove();
                    safeLookaheadPaths.add(Collections.singletonList(lookahead));
                } else {
                    result = true;
                    lookaheadPaths.add(Collections.singletonList(lookahead));
                }
                /**/
            }
            
            if (logger.verboseLevel() < Logger.LEVEL_VERBOSE) {
                safeLookaheadPaths = Collections.EMPTY_SET;
            }
            if (nextSymbols.size() + safeLookaheadPaths.size() > 0) {
                reportConflict(conflictedState, lrHelper, lookaheadPaths, safeLookaheadPaths);
            }
            
            return result;
        }
        
        public List getMaybeConflictPath(LRState lrState, Symbol lookahead) {
            MultiMap labelToLRStates = new MultiMap();
            {
                Iterator it55 = lrState.items.iterator();
                while (it55.hasNext()) {
                    Item item = (Item) it55.next();
                    if (item.core.isFinal) {
                        Reduction reduction = generator.getReduction(item);
                        Set nextLRStates = backstep(lrState, reduction.labelSetList().size());
                        nextLRStates = getNextLRStates(nextLRStates, reduction.reducedSymbol());
                        nextLRStates = getNextLRStates(nextLRStates, lookahead);
                        labelToLRStates.addAll(reduction, nextLRStates);
                    }
                }
                Set nextLRStates = shiftSavingMemory(Collections.singleton(lrState), lookahead);
                labelToLRStates.addAll(null, nextLRStates);
            }
            
            ChasedState initialState = new ChasedState(labelToLRStates);
            Map chasedStateToPath = new HashMap();
            chasedStateToPath.put(initialState, TreeStack.push(TreeStack.EMPTY_STACK, lookahead));
            LinkedList undone = new LinkedList();
            undone.add(initialState);
            while (!undone.isEmpty()) {
                ChasedState cs = (ChasedState) undone.removeFirst();
                if (cs.labelToLRStates.size() > 1) {
                    TreeStack path = (TreeStack) chasedStateToPath.get(cs);
                    List list = new ArrayList(cs.labelToLRStates.values());
                    for (int i = 0; i < list.size(); i++) {
                        Set set1 = (Set) list.get(i);
                        for (int j = i + 1; j < list.size(); j++) {
                            Set set2 = (Set) list.get(j);
                            if (set1.equals(set2)) {
                                // We may have the same route to the lrHelper.finalState() from the conflicted states.
                                return new ArrayList(TreeStack.pushAll(TreeStack.EMPTY_STACK, path));
                            }
                        }
                    }
                    Iterator it62 = cs.symbolToNextState().entrySet().iterator();
                    while (it62.hasNext()) {
                        Map.Entry entry = (Map.Entry) it62.next();
                        Symbol symbol = (Symbol) entry.getKey();
                        ChasedState nextState = (ChasedState) entry.getValue();
                        if (!chasedStateToPath.containsKey(nextState)) {
                            chasedStateToPath.put(nextState, TreeStack.push(path, symbol));
                            undone.add(nextState);
                        }
                    }
                }
            }
            
            return null;    // sure no conflict.
        }
        
        private class ChasedState implements SimpleState {
            public final MultiMap labelToLRStates;
            public final int hashCode;
            public ChasedState(MultiMap labelToLRStates) {
                this.labelToLRStates = labelToLRStates;
                this.hashCode = labelToLRStates.hashCode();
            }
            
            public int hashCode() {
                return hashCode;
            }
            public final boolean equals(Object rhs) {
                if (this == rhs)
                    return true;
                if (rhs instanceof ChasedState)
                    return equals((ChasedState) rhs);
                return false;
            }
            public boolean equals(ChasedState rhs) {
                return hashCode == rhs.hashCode && labelToLRStates.equals(rhs.labelToLRStates);
            }
            
            public String toString() {
                return labelToLRStates.toString();
            }
            
            private Map symbolToNextState = null;
            public Map symbolToNextState() {
                if (symbolToNextState == null) {
                    Map resultMaps = new LinkedHashMap();
                    Iterator it57 = labelToLRStates.entrySet().iterator();
                    while (it57.hasNext()) {
                        Map.Entry entry = (Map.Entry) it57.next();
                        Object label = entry.getKey();
                        Set lrStates = (Set) entry.getValue();
                        lrStates = reduceAllSavingMemory(lrStates, null);
                        Iterator it61 = lrStates.iterator();
                        while (it61.hasNext()) {
                            LRState lrState = (LRState) it61.next();
                            Iterator it65 = lrState.symbolToNextState.entrySet().iterator();
                            while (it65.hasNext()) {
                                Map.Entry entry2 = (Map.Entry) it65.next();
                                Symbol symbol = (Symbol) entry2.getKey();
                                LRState nextLRState = (LRState) entry2.getValue();
                                MultiMap resultMap = (MultiMap) resultMaps.get(symbol);
                                if (resultMap == null) {
                                    resultMap = new MultiMap();
                                    resultMaps.put(symbol, resultMap);
                                }
                                resultMap.add(label, nextLRState);
                            }
                        }
                    }
                    
                    symbolToNextState = new LinkedHashMap();
                    Iterator it49 = resultMaps.entrySet().iterator();
                    while (it49.hasNext()) {
                        Map.Entry entry = (Map.Entry) it49.next();
                        Symbol symbol = (Symbol) entry.getKey();
                        MultiMap resultMap = (MultiMap) entry.getValue();
                        symbolToNextState.put(symbol, new ChasedState(resultMap));
                    }
                }
                return symbolToNextState;
            }
        } // class ChasedState
        
        
        /************************************************************************
         * requres too much resouces.
        private boolean isResolvableConflict(LRState conflictedState, Symbol lookahead) {
            LinkedList undone = new LinkedList();
            Iterator it80 = shift(reduce(conflictedState, null, lookahead, 2), lookahead).iterator();
            while (it80.hasNext()) {
                MultiMap lrStateToConflicts = (MultiMap) it80.next();
                undone.add(new ChaseState(lrStateToConflicts));
            }
            
            Set done = new HashSet();
            while (!undone.isEmpty()) {
                ChaseState state = (ChaseState) undone.removeFirst();
                if (done.add(state)) {
                    System.out.println("resolvable? "  + state);
                    Set conflicts = new HashSet();
                    Iterator it81 = state.lrStateToConflicts.entrySet().iterator();
                    while (it81.hasNext()) {
                        Map.Entry entry = (Map.Entry) it81.next();
                        LRState s = (LRState) entry.getKey();
                        Set set = (Set) entry.getValue();
                        if (set.size() > 1) {
                            // We may have the same route to the final state from the conflicted states.
                            // TODO: reports the route to users (in the regular expression perhaps).
                            return false;
                        }
                        conflicts.addAll(set);
                    }
                    
                    if (conflicts.size() > 0) {
                        MultiMap symbolToNextStates = state.symbolToNextStates();
                        System.out.println("resolvable? symbolToNextStates "  + symbolToNextStates);
                        Iterator it82 = symbolToNextStates.values().iterator();
                        while (it82.hasNext()) {
                            Set states = (Set) it82.next();
                            undone.addAll(states);
                        }
                    }
                }
            }
            return true;
        }
        
        private class ChaseState {
            public final MultiMap lrStateToConflicts;
            public ChaseState(MultiMap lrStateToConflicts) {
                this.lrStateToConflicts = lrStateToConflicts;
            }
            
            private MultiMap symbolToNextStates = null;
            public MultiMap symbolToNextStates() {
                if (symbolToNextStates == null) {
                    Set symbols = new LinkedHashSet();
                    Iterator it70 = lrStateToConflicts.keySet().iterator();
                    while (it70.hasNext()) {
                        LRState state = (LRState) it70.next();
                        symbols.addAll(state.symbolToReductions.keySet());
                        symbols.addAll(state.symbolToNextState.keySet());
                    }
                    symbolToNextStates = new MultiMap();
                    Iterator it75 = symbols.iterator();
                    while (it75.hasNext()) {
                        Symbol symbol = (Symbol) it75.next();
                        if (symbol instanceof Terminal) {
                            Set set = reduce(lrStateToConflicts, symbol);
                            set = shift(set, symbol);
                            Iterator it76 = set.iterator();
                            while (it76.hasNext()) {
                                MultiMap lrStateToConflicts = (MultiMap) it76.next();
                                symbolToNextStates.add(symbol, new ChaseState(lrStateToConflicts));
                            }
                        }
                    }
                }
                return symbolToNextStates;
            }
            
            public String toString() {
                return lrStateToConflicts.toString();
            }
        }
        private Set shift(Set multiMaps, Symbol symbol) {
            Set result = new LinkedHashSet(multiMaps.size());
            Iterator it78 = multiMaps.iterator();
            while (it78.hasNext()) {
                MultiMap lrStateToConflicts = (MultiMap) it78.next();
                result.add(shift(lrStateToConflicts, symbol));
            }
            return result;
        }
        private MultiMap shift(MultiMap lrStateToConflicts, Symbol symbol) {
            MultiMap result = new MultiMap();
            Iterator it77 = lrStateToConflicts.entrySet().iterator();
            while (it77.hasNext()) {
                Map.Entry entry = (Map.Entry) it77.next();
                LRState state = (LRState) entry.getKey();
                Set conflicts = (Set) entry.getValue();
                
                state = (LRState) state.symbolToNextState.get(symbol);
                if (state != null)
                    result.addAll(state, conflicts);
            }
            return result;
        }
        private Set reduce(MultiMap lrStateToConflicts, Symbol lookahead) {
            MultiMap base = new MultiMap();
            List results = new ArrayList(lrStateToConflicts.size());
            Iterator it69 = lrStateToConflicts.entrySet().iterator();
            while (it69.hasNext()) {
                Map.Entry entry = (Map.Entry) it69.next();
                LRState state = (LRState) entry.getKey();
                Set conflicts = (Set) entry.getValue();
                
                Set reducedMultiMaps = reduce(state, conflicts, lookahead, 1);
                if (reducedMultiMaps.isEmpty()) {
                } else if (reducedMultiMaps.size() == 1) {
                    MultiMap map = (MultiMap) reducedMultiMaps.iterator().next();
                    base.merge(map);
                } else {
                    results.add(new ArrayList(reducedMultiMaps));
                }
            }
            
            if (results.isEmpty()) {
                return Collections.singleton(base);
            }
            
            Set result = new HashSet();
            int indexes[] = new int[results.size()];
        loop:
            for (;;) {
                MultiMap subResult = new MultiMap(base);
                for (int i = 0; i < indexes.length; i++) {
                    List list = (List) results.get(i);
                    MultiMap map = (MultiMap) list.get(indexes[i]);
                    subResult.merge(map);
                }
                result.add(subResult);
                for (int i = 0; i < indexes.length; i++) {
                    List list = (List) results.get(i);
                    if (++indexes[i] < list.size())
                        continue loop;
                    indexes[i] = 0;
                }
                break loop;
            }
            return result;
        }
        private Set reduce(LRState finalState, Set givenConflicts, Symbol lookahead, int depthThresholdToSaveMemory) {
            LinkedList undone = new LinkedList();
            if (givenConflicts == null) {
                MultiMap initialMarkers = new MultiMap();
                Set reductions = finalState.symbolToReductions.getValueSet(lookahead);
                if (reductions != null) {
                    Iterator it79 = reductions.iterator();
                    while (it79.hasNext()) {
                        Reduction reduction = (Reduction) it79.next();
                        ChaseMarker cm = new ChaseMarker(reduction.reducedSymbol(), reduction.labelSetList().size());
                        initialMarkers.add(cm, reduction);
                    }
                }
                
                MultiMap fixedMap = new MultiMap();
                if (finalState.symbolToNextState.containsKey(lookahead)) {
                    fixedMap.add(finalState, null);
                }
                
                undone.add(new ChaseJob(finalState, initialMarkers, fixedMap));
            } else {
                MultiMap initialMap = new MultiMap();
                initialMap.addAll(finalState, givenConflicts);
                
                Set reductions = finalState.symbolToReductions.getValueSet(lookahead);
                if (reductions == null)
                    return Collections.singleton(initialMap);
                
                MultiMap initialMarkers = new MultiMap();
                Iterator it71 = reductions.iterator();
                while (it71.hasNext()) {
                    Reduction reduction = (Reduction) it71.next();
                    ChaseMarker cm = new ChaseMarker(reduction.reducedSymbol(), reduction.labelSetList().size());
                    initialMarkers.addAll(cm, givenConflicts);
                }
                
                undone.add(new ChaseJob(finalState, initialMarkers, initialMap));
            }
            
            Set result = new LinkedHashSet();
            Map scheduledJobToDepth = new HashMap();
            MultiMap stateToJobs = new MultiMap();
            Iterator it84 = undone.iterator();
            while (it84.hasNext()) {
                ChaseJob job = (ChaseJob) it84.next();
                scheduledJobToDepth.put(job, new Integer(0));
                stateToJobs.add(job.state, job);
            }
            
            while (!undone.isEmpty()) {
                ChaseJob job = (ChaseJob) undone.removeFirst();
                Integer depthInteger = (Integer) scheduledJobToDepth.get(job);
                int depth = depthInteger.intValue();
                if (undone.size() % 100 == 0)
                    System.out.println(undone.size() + " " + scheduledJobToDepth.size() + " " + depth + " " + job.markerToConflicts.size() + " " + job.fixedMap.size());
                if (depth >= depthThresholdToSaveMemory) {
                    reduceSavingMemory(result, job, lookahead);
                    continue;
                }
                
            loop:
                for (;;) {
                    if (job.markerToConflicts.isEmpty()) {
                        result.add(job.fixedMap);
                    } else {
                        MultiMap newMarkers = new MultiMap();
                        MultiMap newFixedMap = null;
                        Iterator it72 = job.markerToConflicts.entrySet().iterator();
                        while (it72.hasNext()) {
                            Map.Entry entry = (Map.Entry) it72.next();
                            ChaseMarker marker = (ChaseMarker) entry.getKey();
                            Set conflicts = (Set) entry.getValue();
                            if (marker.count == 0) {
                                if (newFixedMap == null)
                                    newFixedMap = new MultiMap(job.fixedMap);
                                
                                TreeStack stack = TreeStack.push(TreeStack.EMPTY_STACK, job.state);
                                ChaseMarker seedMarker = new ChaseMarker(stack, marker.reducedSymbol, 0);
                                shiftBack(conflicts, newFixedMap, newMarkers, seedMarker, lookahead);
                            } else {
                                ChaseMarker cm = new ChaseMarker(marker.reducedSymbol, marker.count - 1);
                                newMarkers.addAll(cm, conflicts);
                            }
                        }
                        if (newFixedMap == null)
                            newFixedMap = job.fixedMap;
                        if (newMarkers.isEmpty()) {
                            result.add(newFixedMap);
                        } else {
                            Set backStates = lrHelper.backstep(job.state);
                            if (backStates.size() == 1) {
                                LRState state = (LRState) backStates.iterator().next();
                                job = new ChaseJob(state, newMarkers, newFixedMap);
                                continue loop;
                            }
                            System.out.println(backStates.size());
                            Iterator it74 = backStates.iterator();
                            while (it74.hasNext()) {
                                LRState state = (LRState) it74.next();
                                ChaseJob j = new ChaseJob(state, newMarkers, newFixedMap);
                                if (!scheduledJobToDepth.containsKey(j)) {
                                    boolean nonsense = false;
                                    Set seniorJobs = stateToJobs.getValueSet(j.state);
                                    if (seniorJobs != null) {
                                        Iterator it64 = seniorJobs.iterator();
                                        while (it64.hasNext()) {
                                            ChaseJob seniorJob = (ChaseJob) it64.next();
                                            if (seniorJob.covers(job)) {
                                                nonsense = true;
                                                break;
                                            } else if (job.covers(seniorJob)) {
                                                it64.remove();
                                            }
                                        }
                                    }
                                    if (!nonsense) {
                                        scheduledJobToDepth.put(j, new Integer(depth + 1));
                                        undone.addLast(j);
                                        stateToJobs.add(j.state, j);
                                    }
                                }
                            }
                        }
                    }
                    break loop;
                }
            }
            
            return result;
        }
        private void shiftBack(Set conflicts, MultiMap newFixedMap, MultiMap newMarkers, ChaseMarker seed, Symbol lookahead) {
            LinkedList undone = new LinkedList();
            undone.add(seed);
            Set done = new HashSet();
            while (!undone.isEmpty()) {
                ChaseMarker marker = (ChaseMarker) undone.removeFirst();
                if (done.add(marker)) {
                    if (marker.count >= marker.stack.size()) {
                        marker = new ChaseMarker(marker.reducedSymbol, marker.count - marker.stack.size());
                        newMarkers.addAll(marker, conflicts);
                    } else {
                        TreeStack stack = marker.stack;
                        for (int i = 0; i < marker.count; i++)
                            stack = TreeStack.pop(stack);
                        LRState state = (LRState) stack.top();
                        LRState shiftedState = (LRState) state.symbolToNextState.get(marker.reducedSymbol);
                        newFixedMap.addAll(shiftedState, conflicts);
                        stack = TreeStack.push(stack, shiftedState);
                        Set reductions = shiftedState.symbolToReductions.getValueSet(lookahead);
                        if (reductions != null) {
                            Iterator it73 = reductions.iterator();
                            while (it73.hasNext()) {
                                Reduction reduction = (Reduction) it73.next();
                                undone.add(new ChaseMarker(stack, reduction.reducedSymbol(), reduction.labelSetList().size()));
                            }
                        }
                    }
                }
            }
        }
        private static class ChaseJob {
            public final LRState state;
            public final MultiMap markerToConflicts;
            public final MultiMap fixedMap;
            public ChaseJob(LRState state, MultiMap markerToConflicts, MultiMap fixedMap) {
                this.state = state;
                this.markerToConflicts = markerToConflicts;
                this.fixedMap = fixedMap;
            }
            
            public int hashCode() {
                return state.hashCode() + markerToConflicts.hashCode() + fixedMap.hashCode();
            }
            public final boolean equals(Object rhs) {
                if (this == rhs)
                    return true;
                if (rhs instanceof ChaseJob)
                    return equals((ChaseJob) rhs);
                return false;
            }
            public boolean equals(ChaseJob rhs) {
                return state.equals(rhs.state) && markerToConflicts.equals(rhs.markerToConflicts) && fixedMap.equals(rhs.fixedMap);
            }
            
            public boolean covers(ChaseJob job) {
                if (!state.equals(job.state))
                    return false;
                
                Iterator it66 = job.fixedMap.entrySet().iterator();
                while (it66.hasNext()) {
                    Map.Entry entry = (Map.Entry) it66.next();
                    LRState state = (LRState) entry.getKey();
                    Set conflicts = (Set) entry.getValue();
                    Set thisConflicts = fixedMap.getValueSet(state);
                    if (thisConflicts == null)
                        thisConflicts = Collections.EMPTY_SET;
                    if (!thisConflicts.containsAll(conflicts))
                        return false;
                }
                
                Iterator it67 = job.markerToConflicts.entrySet().iterator();
                while (it67.hasNext()) {
                    Map.Entry entry = (Map.Entry) it67.next();
                    ChaseMarker marker = (ChaseMarker) entry.getKey();
                    Set conflicts = (Set) entry.getValue();
                    Set thisConflicts = markerToConflicts.getValueSet(marker);
                    if (thisConflicts == null)
                        thisConflicts = Collections.EMPTY_SET;
                    if (!thisConflicts.containsAll(conflicts))
                        return false;
                }
                
                return true;
            }
        }
        private static class ChaseMarker {
            public final TreeStack stack;
            public final Symbol reducedSymbol;
            public final int count;
            public ChaseMarker(Symbol reducedSymbol, int count) {
                this(TreeStack.EMPTY_STACK, reducedSymbol, count);
            }
            public ChaseMarker(TreeStack stack, Symbol reducedSymbol, int count) {
                this.stack = stack;
                this.reducedSymbol = reducedSymbol;
                this.count = count;
            }
            public int hashCode() {
                return stack.hashCode() + reducedSymbol.hashCode() + count;
            }
            public final boolean equals(Object rhs) {
                if (this == rhs)
                    return true;
                if (rhs instanceof ChaseMarker)
                    return equals((ChaseMarker) rhs);
                return false;
            }
            public boolean equals(ChaseMarker rhs) {
                return stack.equals(rhs.stack) && reducedSymbol.equals(rhs.reducedSymbol) && count == rhs.count;
            }
            
            public String toString() {
                return stack + " " + reducedSymbol + " " + count;
            }
        }
        
        public void reduceSavingMemory(Set result, ChaseJob job, Symbol lookahead) {
            MultiMap conflictToMarkers = new MultiMap();
            Iterator it83 = job.markerToConflicts.entrySet().iterator();
            while (it83.hasNext()) {
                Map.Entry entry = (Map.Entry) it83.next();
                ChaseMarker marker = (ChaseMarker) entry.getKey();
                Set conflicts = (Set) entry.getValue();
                Iterator it85 = conflicts.iterator();
                while (it85.hasNext()) {
                    Reduction conflict = (Reduction) it85.next();
                    conflictToMarkers.add(conflict, marker);
                }
            }
            
            MultiMap fixedMap = new MultiMap(job.fixedMap);
            Iterator it86 = conflictToMarkers.entrySet().iterator();
            while (it86.hasNext()) {
                Map.Entry entry = (Map.Entry) it86.next();
                Reduction conflict = (Reduction) entry.getKey();
                Set markers = (Set) entry.getValue();
                Set lrStates = reduceSavingMemory(conflict, job.state, markers, lookahead);
                Iterator it87 = lrStates.iterator();
                while (it87.hasNext()) {
                    LRState state = (LRState) it87.next();
                    fixedMap.add(state, conflict);
                }
            }
            
            result.add(fixedMap);
        }
        private Set reduceSavingMemory(Reduction conflict, LRState initialState, Set markers, Symbol lookahead) {
            Set lrStates = new LinkedHashSet();
            Iterator it88 = markers.iterator();
            while (it88.hasNext()) {
                ChaseMarker marker = (ChaseMarker) it88.next();
                Set nextStates = backstep(initialState, marker.count);
                nextStates = getNextLRStatesPrivate(Collections.singleton(initialState), marker.reducedSymbol);
                lrStates.addAll(nextStates);
            }
            return reduceAllSavingMemory(lrStates, lookahead);
        }
        /************************************************************************/
        
        private Set backstep(LRState lrState, int count) {
            return backstep(Collections.singleton(lrState), count);
        }
        private Set backstep(Set lrStates, int count) {
            if (count == 0)
                return lrStates;
            
            if (lrStates.isEmpty())
                return Collections.EMPTY_SET;
            
            Set result = new LinkedHashSet();
            Iterator it60 = lrStates.iterator();
            while (it60.hasNext()) {
                LRState lrState = (LRState) it60.next();
                result.addAll(lrHelper.backstep(lrState));
            }
            return backstep(result, count - 1);
        }
        
        private Map nextLRStatesCache = new HashMap();
        private Set getNextLRStates(Set lrStates, Symbol nextSymbol) {
            List key = Arrays.asList(new Object[] { lrStates, nextSymbol });
            Set result = (Set) nextLRStatesCache.get(key);
            if (result == null) {
                result = getNextLRStatesPrivate(lrStates, nextSymbol);
                nextLRStatesCache.put(key, result);
            }
            return result;
        }
        private Map nextLRStatesBusy = new HashMap();
        private Set getNextLRStatesPrivate(Set lrStates, Symbol nextSymbol) {
            List key = Arrays.asList(new Object[] { lrStates, nextSymbol });
            Set result = (Set) nextLRStatesCache.get(key);
            if (result != null) {
                return result;
            }
            
            result = (Set) nextLRStatesBusy.get(key);
            if (result != null)
                return result;
            
            nextLRStatesBusy.put(key, Collections.EMPTY_SET);
            try {
                return shiftSavingMemory(reduceAllSavingMemory(lrStates, nextSymbol), nextSymbol);
            } finally {
                nextLRStatesBusy.remove(key);
            }
        }
        private Set reduceAllSavingMemory(Set lrStates, Symbol lookahead) {
            Set result = new LinkedHashSet();
            LinkedList undone = new LinkedList(lrStates);
            while (!undone.isEmpty()) {
                LRState lrState = (LRState) undone.removeFirst();
                if (result.add(lrState)) {
                    if (lookahead != null) {
                        Set reductions = (Set) lrState.symbolToReductions.get(lookahead);
                        if (reductions != null) {
                            Iterator it56 = reductions.iterator();
                            while (it56.hasNext()) {
                                Reduction reduction = (Reduction) it56.next();
                                Set nextStates = backstep(lrState, reduction.labelSetList().size());
                                nextStates = getNextLRStatesPrivate(nextStates, reduction.reducedSymbol());
                                undone.addAll(nextStates);
                            }
                        }
                    } else {
                        Iterator it5 = lrState.items.iterator();
                        while (it5.hasNext()) {
                            Item item = (Item) it5.next();
                            if (item.core.isFinal) {
                                Reduction reduction = generator.getReduction(item);
                                Set nextStates = backstep(lrState, reduction.labelSetList().size());
                                nextStates = getNextLRStatesPrivate(nextStates, reduction.reducedSymbol());
                                undone.addAll(nextStates);
                            }
                        }
                    }
                }
            }
            return result;
        }
        private Set shiftSavingMemory(Set lrStates, Symbol lookahead) {
            Set result = new LinkedHashSet();
            Iterator it58 = lrStates.iterator();
            while (it58.hasNext()) {
                LRState lrState = (LRState) it58.next();
                LRState nextLRState = (LRState) lrState.symbolToNextState.get(lookahead);
                if (nextLRState != null)
                    result.add(nextLRState);
            }
            return result;
        }
        
        
        private void reportConflict(LRState state, LRHelper lrHelper, Set lookaheadPaths, Set safelookaheadPaths) {
            List path = getExamplePath(state, lrHelper, initialStateToSymbol);
            TreeStack stack = TreeStack.pushAll(TreeStack.EMPTY_STACK, path);
            
            Iterator it = path.iterator();
            Nonterminal nonterminal = (Nonterminal) it.next();
            it.next();  // state
            StringBuffer buffer = new StringBuffer();
            while (it.hasNext()) {
                Symbol symbol = (Symbol) it.next();
                it.next();  // state
                if (symbol instanceof AnonymousNonterminal)
                    buffer.append("...");
                else
                    buffer.append(symbol);
                if (it.hasNext())
                    buffer.append(' ');
            }
            
            List paths = new ArrayList(lookaheadPaths.size() + safelookaheadPaths.size());
            paths.addAll(lookaheadPaths);
            paths.addAll(safelookaheadPaths);
            
            int level = !lookaheadPaths.isEmpty() ? Logger.LEVEL_WARNING : Logger.LEVEL_VERBOSE;
            Token hint = getHint(nonterminal);
            
            logger.report(level, hint, "LALR(1) conflicts: One of the input to the conflicted state is");
            logger.report(level, hint, "        {1}, to parse {0}.", hint, buffer);
            /**** requires too much resources.
            logger.verbose(hint, "The all input(s) to the conflicted state are the follwings:");
            Iterator it27 = lrHelper.symbolToInitialLRState.entrySet().iterator();
            while (it27.hasNext()) {
                Map.Entry entry = (Map.Entry) it27.next();
                Nonterminal symbol = (Nonterminal) entry.getKey();
                LRState s = (LRState) entry.getValue();
                String re = getPathInRegularExpression(s, state);
                if (re != null)
                    logger.verbose(getHint(symbol), "        {0} with {1}", symbol, re);
            }*/
            for (int i = 0; i < paths.size(); i++) {
                List lookaheadPath = (List) paths.get(i);
                Symbol lookahead = (Symbol) lookaheadPath.get(0);
                level = (i < lookaheadPaths.size()) ? Logger.LEVEL_WARNING : Logger.LEVEL_VERBOSE;
                
                logger.report(level, hint, "    For the look-ahead {0},", lookahead);
                if (lookaheadPath.size() > 1) {
                    StringBuffer lookaheadPathString = new StringBuffer();
                    Iterator it63 = lookaheadPath.iterator();
                    while (it63.hasNext()) {
                        Symbol s = (Symbol) it63.next();
                        if (s instanceof AnonymousNonterminal)
                            lookaheadPathString.append("...");
                        else
                            lookaheadPathString.append(s);
                        if (it63.hasNext())
                            lookaheadPathString.append(' ');
                    }
                    logger.verbose(hint, "        (or {0},)", lookaheadPathString);
                }
                logger.report(level, hint, "    the reducible nonterminals are");
                boolean shiftReduceConflict = state.symbolToNextState.containsKey(lookahead);
                
                Iterator it28 = state.symbolToReductions.getValueSet(lookahead).iterator();
                while (it28.hasNext()) {
                    Reduction reduction = (Reduction) it28.next();
                    TreeStack tail = TreeStack.EMPTY_STACK;
                    reportConflictReduction(level, stack, reduction, 0, tail, lookahead);
                }
                
                if (shiftReduceConflict) {
                    logger.report(level, hint, "        also shiftable.");
                }
            }
        }
        
        /*private void reportShiftConflicts(TreeStack stack, LRState state, Symbol lookahead) {
            TreeStack tail = TreeStack.push(TreeStack.EMPTY_STACK, null);
            stack = TreeStack.push(stack, lookahead);
            
            Set nextItems = new LinkedHashSet();
            Iterator it53 = state.items.iterator();
            while (it53.hasNext()) {
                Item item = (Item) it53.next();
                Set set = item.symbolToNextItems.getValueSet(lookahead);
                if (set != null)
                    nextItems.addAll(set);
            }
            
            Iterator it54 = nextItems.iterator();
            while (it54.hasNext()) {
                Item item = (Item) it54.next();
                reportConflictReduction(TreeStack.push(stack, item.owner), getReduction(item), 0, tail, null);
            }
        }*/
        private void reportConflictReduction(int level, TreeStack stack, Reduction reduction, int ignoreCount, TreeStack tail, Symbol lookahead) {
            Nonterminal reducedSymbol = reduction.reducedSymbol();
            List labelSetList = reduction.labelSetList();
            for (int i = labelSetList.size() - 1 - ignoreCount; i >= 0; i--) {
                stack = TreeStack.pop(stack);   // state
                Symbol symbol = (Symbol) stack.top();
                stack = TreeStack.pop(stack);
                
                tail = TreeStack.push(tail, symbol);
                tail = TreeStack.push(tail, labelSetList.get(i));
            }
            
            if (reducedSymbol instanceof AnonymousNonterminal) {
                LRState state = (LRState) stack.top();
                state = (LRState) state.symbolToNextState.get(reducedSymbol);
                reportConflictReduction(level, stack, state, tail, lookahead);
            } else {
                StringBuffer buffer = new StringBuffer();
                if (tail.isEmpty()) {
                    buffer.append("(empty)");
                } else {
                    Iterator it45 = tail.iterator();
                    while (it45.hasNext()) {
                        Set labelSet = (Set) it45.next();
                        if (labelSet == null) {
                            buffer.append("...");
                            assert !it45.hasNext();
                            break;
                        } else {
                            Symbol symbol = (Symbol) it45.next();
                            
                            Iterator it22 = labelSet.iterator();
                            while (it22.hasNext()) {
                                String label = (String) it22.next();
                                if (!label.equals("$label")) {
                                    buffer.append(label);
                                    buffer.append(":");
                                }
                            }
                            if (symbol instanceof AnonymousNonterminal)
                                buffer.append("...");
                            else
                                buffer.append(symbol);
                            if (it45.hasNext())
                                buffer.append(' ');
                        }
                    }
                }
                Token hint = getHint(reducedSymbol);
                logger.report(level, "        {0} from {1}", hint, buffer);
            }
        }
        private void reportConflictReduction(int level, TreeStack stack, LRState state, TreeStack tail, Symbol lookahead) {
            Set reductions;
            if (lookahead != null) {
                reductions = state.symbolToReductions.getValueSet(lookahead);
            } else {
                reductions = new LinkedHashSet();
                Iterator it42 = state.symbolToReductions.values().iterator();
                while (it42.hasNext()) {
                    Set set = (Set) it42.next();
                    reductions.addAll(set);
                }
            }
            Iterator it28 = reductions.iterator();
            while (it28.hasNext()) {
                Reduction reduction = (Reduction) it28.next();
                reportConflictReduction(level, stack, reduction, 1, tail, lookahead);
            }
        }
    }
    
    private LRTable buildLRTable(LRHelper lrHelper) {
        return new GeneratedLRTable(lrHelper.symbolToInitialLRState(), lrHelper.finalState());
    }
    private static class GeneratedLRTable extends LRTable {
        private final Map symbolToInitialState;
        private final LRState finalState;
        public GeneratedLRTable(Map symbolToInitialState, LRState finalState) {
            this.symbolToInitialState = Collections.unmodifiableMap(symbolToInitialState);
            this.finalState = finalState;
        }
        
        public Map symbolToInitialState() {
            return symbolToInitialState;
        }
        public LRState finalState() {
            return finalState;
        }
    }
    
    private static MultiMap getBackstep(SimpleState initialState) {
        MultiMap result = new MultiMap();
        LinkedList undone = new LinkedList();
        undone.add(initialState);
        Set done = new HashSet();
        while (!undone.isEmpty()) {
            SimpleState state = (SimpleState) undone.removeFirst();
            if (done.add(state)) {
                Iterator it23 = state.symbolToNextState().values().iterator();
                while (it23.hasNext()) {
                    SimpleState s = (SimpleState) it23.next();
                    result.add(s, state);
                    undone.add(s);
                }
            }
        }
        return result;
    }
    
    /*private static class RegularExpression {
        private Set repeatableExpressions;
        private Set unrepeatableExpressions;
        public RegularExpression(Set repeatableExpressions, Set unrepeatableExpressions) {
            this.repeatableExpressions = repeatableExpressions;
            this.unrepeatableExpressions = unrepeatableExpressions;
        }
        
        public int hashCode() {
            return repeatableExpressions.hashCode() + unrepeatableExpressions.hashCode();
        }
        public final boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (rhs instanceof RegularExpression)
                return equals((RegularExpression) rhs);
            return false;
        }
        public boolean equals(RegularExpression rhs) {
            return repeatableExpressions.equals(rhs.repeatableExpressions) + unrepeatableExpressions.equals(rhs.unrepeatableExpressions);
        }
        
        public String toString() {
            StringBuffer result = new StringBuffer();
            toString(result);
            return result;
        }
        private void toString(StringBuffer result) {
            if (!repeatableExpressions.isEmpty()) {
                toString(result, repeatableExpressions);
                result.append('*');
                if (!unrepeatableExpressions.isEmpty())
                    result.append(' ');
            }
            if (!unrepeatableExpressions.isEmpty()) {
                toString(result, unrepeatableExpressions);
            }
        }
        private void toString(StringBuffer result, Set lists) {
            boolean paren = true;
            if (lists.size() == 1) {
                List list = (List) lists.iterator().next();
                if (list.size() == 1) {
                    paren = false;
                }
            }
            if (paren)
                result.append("( ");
            
            Iterator outer = lists.iterator().iterator();
            while (outer.hasNext()) {
                List list = (List) outer.next();
                Iterator inner = list.iterator();
                while (inner.hasNext()) {
                    Object o = (Object) inner.next();
                    if (o instanceof RegularExpression) {
                        RegularExpression re = (RegularExpression) o;
                        re.toString(result);
                    } else {
                        result.append(o);
                    }
                    if (inner.hasNext())
                        result.append(' ');
                }
                if (outer.hasNext())
                    result.append(" | ");
            }
            
            if (paren)
                result.append(" )");
        }
    }*/
    private static String getPathInRegularExpression(SimpleState startState, SimpleState lastState) {
        MultiMap backstep = getBackstep(startState);
        Set allStates = new HashSet();
        {
            LinkedList undone = new LinkedList();
            undone.add(lastState);
            while (!undone.isEmpty()) {
                SimpleState state = (SimpleState) undone.removeFirst();
                if (allStates.add(state)) {
                    Set set = backstep.getValueSet(state);
                    if (set != null)
                        undone.addAll(set);
                }
            }
        }
        if (!allStates.contains(startState))
            return null;
        
        LinkedList undone = new LinkedList();
        undone.add(TreeStack.push(TreeStack.EMPTY_STACK, startState));
        MultiMap stateToRepeatableRoutes = new MultiMap();
        
        Set routes = new LinkedHashSet();
        while (!undone.isEmpty()) {
            TreeStack route = (TreeStack) undone.removeFirst();
            System.out.println(route);
            SimpleState state = (SimpleState) route.top();
            if (state.equals(lastState))
                routes.add(TreeStack.pop(TreeStack.pushAll(TreeStack.EMPTY_STACK, route)));
            Iterator it68 = state.symbolToNextState().entrySet().iterator();
            while (it68.hasNext()) {
                Map.Entry entry = (Map.Entry) it68.next();
                Symbol symbol = (Symbol) entry.getKey();
                SimpleState nextState = (SimpleState) entry.getValue();
                if (allStates.contains(nextState)) {
                    if (route.contains(nextState)) {
                        TreeStack stack = TreeStack.push(TreeStack.EMPTY_STACK, nextState);
                        Iterator it = route.iterator();
                        for (;;) {
                            SimpleState s = (SimpleState) it.next();
                            if (s.equals(nextState))
                                break;
                            TreeStack.push(stack, s);
                        }
                        stateToRepeatableRoutes.add(nextState, stack);
                    } else {
                        undone.add(TreeStack.push(route, nextState));
                    }
                }
            }
        }
        
        StringBuffer result = new StringBuffer();
        getPathInRegularExpression(result, TreeStack.EMPTY_STACK, startState, routes, stateToRepeatableRoutes);
        return result.toString();
    }
    private static void getPathInRegularExpression(StringBuffer result, TreeStack stack, SimpleState state, Set routes, MultiMap stateToRepeatableRoutes) {
        if (stack.contains(state))
            return;
        
        stack = TreeStack.push(stack, state);
        System.out.println(state);
        assert !routes.isEmpty();
        Set repeatableRoutes = stateToRepeatableRoutes.getValueSet(state);
        if (repeatableRoutes != null) {
            result.append("( ");
            getPathInRegularExpressionNoRepeat(result, stack, state, repeatableRoutes, stateToRepeatableRoutes);
            result.append(")* ");
        }
        getPathInRegularExpressionNoRepeat(result, stack, state, routes, stateToRepeatableRoutes);
    }
    private static void getPathInRegularExpressionNoRepeat(StringBuffer result, TreeStack stack, SimpleState state, Set routes, MultiMap stateToRepeatableRoutes) {
        boolean hitEmpty = false;
        MultiMap nextStateToRoutes = new MultiMap();
        Iterator it40 = routes.iterator();
        while (it40.hasNext()) {
            TreeStack route = (TreeStack) it40.next();
            if (route.isEmpty()) {
                hitEmpty = true;
            } else {
                SimpleState s = (SimpleState) route.top();
                nextStateToRoutes.add(s, route);
            }
        }
        
        if (nextStateToRoutes.isEmpty())
            return;
        
        MultiMap nextStateToSymbols = new MultiMap();
        Iterator it90 = state.symbolToNextState().entrySet().iterator();
        while (it90.hasNext()) {
            Map.Entry entry = (Map.Entry) it90.next();
            Symbol symbol = (Symbol) entry.getKey();
            SimpleState nextState = (SimpleState) entry.getValue();
            nextStateToSymbols.add(nextState, symbol);
        }
        
        if (hitEmpty)
            result.append("[ ");
        else if (nextStateToSymbols.size() > 1)
            result.append("( ");
        
        Iterator outer = nextStateToRoutes.entrySet().iterator();
        while (outer.hasNext()) {
            Map.Entry entry = (Map.Entry) outer.next();
            SimpleState nextState = (SimpleState) entry.getKey();
            Set nextRoutes = (Set) entry.getValue();
            Set symbols = nextStateToSymbols.getValueSet(nextState);
            
            if (symbols.size() > 1)
                result.append("( ");
            Iterator inner = symbols.iterator();
            while (inner.hasNext()) {
                Symbol symbol = (Symbol) inner.next();
                result.append(symbol);
                if (inner.hasNext())
                    result.append(" | ");
            }
            if (symbols.size() > 1)
                result.append(" )");
            result.append(' ');
            
            Set subRoutes = new LinkedHashSet();
            Iterator it92 = nextRoutes.iterator();
            while (it92.hasNext()) {
                TreeStack route = (TreeStack) it92.next();
                SimpleState s = (SimpleState) route.top();
                subRoutes.add(TreeStack.pop(route));
            }
            getPathInRegularExpression(result, stack, nextState, subRoutes, stateToRepeatableRoutes);
            
            if (outer.hasNext())
                result.append(" | ");
        }
        
        if (hitEmpty)
            result.append(" ]");
        else if (nextStateToSymbols.size() > 1)
            result.append(" )");
    }
    
    
    private static List getExamplePath(LRState state, LRHelper lrHelper, Map initialStateToSymbol) {
        LinkedList result = new LinkedList();
        Map stateToAscendedStateAndSymbol = lrHelper.stateToAscendedStateAndSymbol();
        
        LRState focusedState = state;
        Set focusedKernelItems = focusedState.kernelItems;
        
        boolean beamed = false;
    loop:
        for (;;) {
            //result.addFirst(focusedKernelItems);
            result.addFirst(focusedState);
            
            {
                Symbol symbol = (Symbol) lrHelper.initialLRStateToSymbol().get(focusedState);
                if (symbol != null) {
                    result.addFirst(symbol);
                    break loop;
                }
            }
            
            Object[] ascendedStateAndSymbol = (Object[]) stateToAscendedStateAndSymbol.get(focusedState);
            assert ascendedStateAndSymbol != null;
            LRState ascendedState = (LRState) ascendedStateAndSymbol[0];
            Symbol symbol = (Symbol) ascendedStateAndSymbol[1];
            
            result.addFirst(symbol);
            
            Set ascendedItems = new LinkedHashSet();
                // will be { item | item is in ascendedState && item + symbol is in focusedKernelItems }
            Iterator it51 = ascendedState.items.iterator();
            while (it51.hasNext()) {
                Item previousItem = (Item) it51.next();
                Set items = previousItem.symbolToNextItems.getValueSet(symbol);
                if (items != null) {
                    Iterator it52 = items.iterator();
                    while (it52.hasNext()) {
                        Item item = (Item) it52.next();
                        if (focusedKernelItems.contains(item))
                            ascendedItems.add(previousItem);
                    }
                }
            }
            assert !ascendedItems.isEmpty();
            if (ascendedItems.size() == 1)
                beamed = true;
            
            if (beamed) {
                Iterator it43 = ascendedItems.iterator();
                while (it43.hasNext()) {
                    Item item = (Item) it43.next();
                    Symbol s = (Symbol) initialStateToSymbol.get(item.core);
                    if (s instanceof TypeNonterminal || s instanceof AliasNonterminal) {
                        //result.addFirst(ascendedItems);
                        result.addFirst(ascendedState);
                        result.addFirst(s);
                        break loop;
                    }
                }
            }
            
            Iterator it41 = new ArrayList(ascendedItems).iterator();
            while (it41.hasNext()) {
                Item item = (Item) it41.next();
                Set tmp = lrHelper.getGeneratorKernels(item);
                if (tmp != null)
                    ascendedItems.addAll(tmp);
            }
            ascendedItems.retainAll(ascendedState.kernelItems);
            assert !ascendedItems.isEmpty();
            if (ascendedItems.size() == 1)
                beamed = true;
            
            focusedState = ascendedState;
            focusedKernelItems = ascendedItems;
        }
        
        return result;
    }
}

