package jp.gr.java_conf.koto.notavacc.generator;

import java.util.*;
import java.io.*;
import jp.gr.java_conf.koto.io.*;
import jp.gr.java_conf.koto.notavacc.*;
import jp.gr.java_conf.koto.notavacc.util.*;
import jp.gr.java_conf.koto.notavacc.dfa.*;
import jp.gr.java_conf.koto.notavacc.dfa.DeterministicFiniteAutomaton.*;
import jp.gr.java_conf.koto.notavacc.lrg.*;
import jp.gr.java_conf.koto.notavacc.lrg.LRTable.*;
import jp.gr.java_conf.koto.notavacc.types.*;
import jp.gr.java_conf.koto.notavacc.parser.Original.*;
import jp.gr.java_conf.koto.notavacc.parser.Parser.*;
import jp.gr.java_conf.koto.notavacc.parser.Parser.Root;

public abstract class CodeGenerator extends Module {
    private Root root;
    private DeterministicFiniteAutomaton dfa;
    private LRTable lrTable;
    private TypeSystem typeSystem;
    private Tables tables;
    public CodeGenerator(Environment env, Root root, DeterministicFiniteAutomaton dfa, LRTable lrTable, TypeSystem typeSystem) {
        super(env);
        this.root = root;
        this.dfa = dfa;
        this.lrTable = lrTable;
        this.typeSystem = typeSystem;
        this.tables = new Tables();
    }
    
    public void generate(File file, boolean dryRun) {
        try {
            if (!dryRun)
                logger.verbose(environment.processingFile, "writing to the file ''{0}''.", file);
            else
                logger.verbose(environment.processingFile, "dry run: not writing to the file ''{0}''.", file);
            generateCode(file, tables, dryRun);
            if (!dryRun)
                logger.verbose(environment.processingFile, "wrote.  {0} bytes.", new Long(file.length()));
            else
                logger.verbose(environment.processingFile, "dry run: did not write.");
        } catch (IOException x) {
            logger.error(environment.processingFile, "An error occured during writing the file ''{0}'': {1}", file, x.getLocalizedMessage());
            
            StringWriter stringWriter = new StringWriter();
            PrintWriter printWriter = new PrintWriter(stringWriter);
            x.printStackTrace(printWriter);
            printWriter.close();
            assert !printWriter.checkError();
            logger.verbose(environment.processingFile, "{0}", stringWriter.toString());
            
            logger.fatal();
        }
    }
    
    protected abstract void generateCode(File file, Tables tables, boolean dryRun) throws IOException;
    
    
    protected class Tables {
        
        // DFAStates
        
        private DFAState[] dfaStates;
        public DFAState[] dfaStates() {
            if (dfaStates == null) {
                Set result = new LinkedHashSet(dfa.states().size());
                result.add(dfa.initialState());
                result.addAll(dfa.states());
                
                dfaStates = (DFAState[]) result.toArray(new DFAState[result.size()]);
            }
            return dfaStates;
        }
        private Map dfaStateToIndexNumber;
        public int getIndexOfState(DFAState state) {
            if (dfaStateToIndexNumber == null) {
                dfaStateToIndexNumber = new HashMap();
                DFAState[] dfaStates = dfaStates();
                for (int i = 0; i < dfaStates.length; i++)
                    dfaStateToIndexNumber.put(dfaStates[i], new Integer(i));
            }
            Integer result = (Integer) dfaStateToIndexNumber.get(state);
            return result.intValue();
        }
        
        
        // symbols.
        
        private Set whiteTerminals = null;
        public Set whiteTerminals() {
            if (whiteTerminals == null) {
                whiteTerminals = new LinkedHashSet();
                Iterator it0 = root.whiteTokenReservations().iterator();
                while (it0.hasNext()) {
                    WhiteTokenDefinition definition = (WhiteTokenDefinition) it0.next();
                    String name = root.getCanonicalName(definition);
                    whiteTerminals.add(new NamedTerminal(name));
                }
            }
            return whiteTerminals;
        }
        private Set blackTerminals = null;
        public Set blackTerminals() {
            if (blackTerminals == null) {
                blackTerminals = new LinkedHashSet();
                Iterator it0 = root.tokenReservations().iterator();
                while (it0.hasNext()) {
                    TokenReservation reservation = (TokenReservation) it0.next();
                    String name = root.getCanonicalName(reservation);
                    blackTerminals.add(new NamedTerminal(name));
                }
                blackTerminals.addAll(dfa.getTerminals());
                blackTerminals.removeAll(whiteTerminals());
            }
            return blackTerminals;
        }
        private Set typeNonterminals = null;
        public Set typeNonterminals() {
            if (typeNonterminals == null) {
                typeNonterminals = new LinkedHashSet();
                Iterator it0 = root.typeDefinitions().iterator();
                while (it0.hasNext()) {
                    TypeDefinition def = (TypeDefinition) it0.next();
                    String name = root.getCanonicalName(def);
                    typeNonterminals.add(new TypeNonterminal(name));
                }
                assert typeNonterminals.containsAll(lrTable.getTypeNonterminals());
                typeNonterminals.retainAll(lrTable.getTypeNonterminals());
            }
            return typeNonterminals;
        }
        private Set aliasNonterminals;
        public Set aliasNonterminals() {
            if (aliasNonterminals == null) {
                aliasNonterminals = new LinkedHashSet();
                Iterator it0 = root.aliasDefinitions().iterator();
                while (it0.hasNext()) {
                    AliasDefinition def = (AliasDefinition) it0.next();
                    String name = root.getCanonicalName(def);
                    aliasNonterminals.add(new AliasNonterminal(name));
                }
                assert aliasNonterminals.containsAll(lrTable.getAliasNonterminals());
                aliasNonterminals.retainAll(lrTable.getAliasNonterminals());
            }
            return aliasNonterminals;
        }
        public Set anonymousNonterminals() {
            return lrTable.getAnonymousNonterminals();
        }
        
        private Symbol[] symbols = null;
        public Symbol[] symbols() {
            // the result should be ordered: EOF, white token, terminals, type nonterminals, alias noterminals, and anonymous nonterminals.
            
            if (symbols == null) {
                LinkedHashSet result = new LinkedHashSet();
                
                // EOF
                result.add(Terminal.EOF);
                
                // white tokens
                firstIndexOfWhiteTerminals = result.size();
                result.addAll(whiteTerminals());
                
                // terminals
                firstIndexOfBlackTerminals = result.size();
                result.addAll(blackTerminals());
                
                // type nonterminals
                firstIndexOfTypeNonterminals = result.size();
                result.addAll(typeNonterminals());
                
                // alias noterminals
                firstIndexOfAliasNonterminals = result.size();
                result.addAll(aliasNonterminals());
                
                // anonymous nonterminals
                firstIndexOfAnonymousNonterminals = result.size();
                result.addAll(anonymousNonterminals());
                
                symbols = (Symbol[]) result.toArray(new Symbol[result.size()]);
            }
            return symbols;
        }
        private Map symbolToIndexInteger;
        public int getIndexOfSymbol(Symbol symbol) {
            if (symbolToIndexInteger == null) {
                symbolToIndexInteger = new HashMap();
                Symbol[] symbols = symbols();
                for (int i = 0; i < symbols.length; i++)
                    symbolToIndexInteger.put(symbols[i], new Integer(i));
            }
            Integer integer = (Integer) symbolToIndexInteger.get(symbol);
            return integer.intValue();
        }
        
        private int firstIndexOfWhiteTerminals = -1;
        public int firstIndexOfWhiteTerminals() {
            symbols();
            assert (firstIndexOfWhiteTerminals != -1);
            return firstIndexOfWhiteTerminals;
        }
        
        public int superiorIndexOfWhiteTerminals() {
            return firstIndexOfBlackTerminals();
        }
        private int firstIndexOfBlackTerminals = -1;
        public int firstIndexOfBlackTerminals() {
            symbols();
            assert (firstIndexOfBlackTerminals != -1);
            return firstIndexOfBlackTerminals;
        }
        
        public int superiorIndexOfTerminals() {
            return superiorIndexOfBlackTerminals();
        }
        public int superiorIndexOfBlackTerminals() {
            return firstIndexOfTypeNonterminals();
        }
        private int firstIndexOfTypeNonterminals = -1;
        public int firstIndexOfTypeNonterminals() {
            symbols();
            assert (firstIndexOfTypeNonterminals != -1);
            return firstIndexOfTypeNonterminals;
        }
        
        public int superiorIndexOfClassNonterminals() {
            return firstIndexOfAliasNonterminals();
        }
        private int firstIndexOfAliasNonterminals = -1;
        public int firstIndexOfAliasNonterminals() {
            symbols();
            assert (firstIndexOfAliasNonterminals != -1);
            return firstIndexOfAliasNonterminals;
        }
        
        public int superiorIndexOfAliasNonterminals() {
            return firstIndexOfAnonymousNonterminals();
        }
        private int firstIndexOfAnonymousNonterminals = -1;
        public int firstIndexOfAnonymousNonterminals() {
            symbols();
            assert (firstIndexOfAnonymousNonterminals != -1);
            return firstIndexOfAnonymousNonterminals;
        }
        
        public int superiorIndexOfAnonymousNonterminals() {
            return symbols().length;
        }
        
        
        // reductions.
        
        private Reduction[] reductions = null;
        public Reduction[] reductions() {
            if (reductions == null) {
                Set tmp = lrTable.getReductions();
                reductions = (Reduction[]) tmp.toArray(new Reduction[tmp.size()]);
            }
            return reductions;
        }
        private Map reductionToIndexNumber = null;
        public int getIndexOfReduction(Reduction reduction) {
            if (reductionToIndexNumber == null) {
                reductionToIndexNumber = new HashMap();
                Reduction[] reductions = reductions();
                for (int i = 0; i < reductions.length; i++)
                    reductionToIndexNumber.put(reductions[i], new Integer(i));
            }
            Integer result = (Integer) reductionToIndexNumber.get(reduction);
            return result.intValue();
        }
        
        
        // labels.
        
        private String[] labels;
        public String[] labels() {
            if (labels == null) {
                Set tmp = new TreeSet();
                
                Reduction[] reductions = reductions();
                for (int i = 0; i < reductions.length; i++) {
                    Reduction reduction = reductions[i];
                    Iterator it1 = reduction.labelSetList().iterator();
                    while (it1.hasNext()) {
                        Set labelSet = (Set) it1.next();
                        tmp.addAll(labelSet);
                    }
                }
                
                tmp.remove("$label");
                List result = new ArrayList(tmp.size() + 1);
                result.add("$label");
                result.addAll(tmp);
                
                labels = (String[]) result.toArray(new String[result.size()]);
            }
            return labels;
        }
        private Map labelToIndexNumber;
        public int getIndexOfLabel(String label) {
            if (labelToIndexNumber == null) {
                labelToIndexNumber = new HashMap();
                String[] labels = labels();
                for (int i = 0; i < labels.length; i++)
                    labelToIndexNumber.put(labels[i], new Integer(i));
            }
            Integer result = (Integer) labelToIndexNumber.get(label);
            return result.intValue();
        }
        public int labelWordCount() {
            return (labels().length + 31) / 32;
        }
        
        
        // LRStates
        
        private LRState[] lrStates;
        public LRState[] lrStates() {
            if (lrStates == null) {
                Set result = new LinkedHashSet(lrTable.states().size());
                result.add(lrTable.finalState());
                result.addAll(lrTable.symbolToInitialState().values());
                result.addAll(lrTable.states());
                
                lrStates = (LRState[]) result.toArray(new LRState[result.size()]);
            }
            return lrStates;
        }
        private Map lrStateToIndexNumber;
        public int getIndexOfState(LRState state) {
            if (lrStateToIndexNumber == null) {
                lrStateToIndexNumber = new HashMap();
                LRState[] lrStates = lrStates();
                for (int i = 0; i < lrStates.length; i++)
                    lrStateToIndexNumber.put(lrStates[i], new Integer(i));
            }
            Integer result = (Integer) lrStateToIndexNumber.get(state);
            return result.intValue();
        }
        
        private Map lrStateToMapTeminalToNextState = new HashMap();
        public Map getMapTeminalToNextState(LRState state) {
            Map result = (Map) lrStateToMapTeminalToNextState.get(state);
            if (result == null) {
                Map map = new TreeMap();
                Iterator it3 = state.symbolToNextState().entrySet().iterator();
                while (it3.hasNext()) {
                    Map.Entry entry = (Map.Entry) it3.next();
                    Symbol symbol = (Symbol) entry.getKey();
                    LRState s = (LRState) entry.getValue();
                    if (symbol instanceof Terminal) {
                        int index = getIndexOfSymbol(symbol);
                        map.put(new Integer(index), s);
                    }
                }
                result = new LinkedHashMap();
                Iterator it4 = map.entrySet().iterator();
                while (it4.hasNext()) {
                    Map.Entry entry = (Map.Entry) it4.next();
                    Integer index = (Integer) entry.getKey();
                    LRState s = (LRState) entry.getValue();
                    result.put(symbols()[index.intValue()], s);
                }
                lrStateToMapTeminalToNextState.put(state, result);
            }
            return result;
        }
        
        private Map lrStateToMapNonteminalToNextState = new HashMap();
        public Map getMapNonteminalToNextState(LRState state) {
            Map result = (Map) lrStateToMapNonteminalToNextState.get(state);
            if (result == null) {
                Map map = new TreeMap();
                Iterator it3 = state.symbolToNextState().entrySet().iterator();
                while (it3.hasNext()) {
                    Map.Entry entry = (Map.Entry) it3.next();
                    Symbol symbol = (Symbol) entry.getKey();
                    LRState s = (LRState) entry.getValue();
                    if (symbol instanceof Nonterminal) {
                        int index = getIndexOfSymbol(symbol);
                        map.put(new Integer(index), s);
                    }
                }
                result = new LinkedHashMap();
                Iterator it4 = map.entrySet().iterator();
                while (it4.hasNext()) {
                    Map.Entry entry = (Map.Entry) it4.next();
                    Integer index = (Integer) entry.getKey();
                    LRState s = (LRState) entry.getValue();
                    result.put(symbols()[index.intValue()], s);
                }
                lrStateToMapNonteminalToNextState.put(state, result);
            }
            return result;
        }
        
        private Map lrStateToMapSymbolToReduction = new HashMap();
        public Map getMapSymbolToReduction(LRState state) {
            Map result = (Map) lrStateToMapSymbolToReduction.get(state);
            if (result == null) {
                Map map = new TreeMap();
                Iterator it3 = state.symbolToReductions().entrySet().iterator();
                while (it3.hasNext()) {
                    Map.Entry entry = (Map.Entry) it3.next();
                    Symbol symbol = (Symbol) entry.getKey();
                    Set reductions = (Set) entry.getValue();
                    int index = getIndexOfSymbol(symbol);
                    map.put(new Integer(index), reductions);
                }
                result = new LinkedHashMap();
                Iterator it4 = map.entrySet().iterator();
                while (it4.hasNext()) {
                    Map.Entry entry = (Map.Entry) it4.next();
                    Integer index = (Integer) entry.getKey();
                    Set reductions = (Set) entry.getValue();
                    result.put(symbols()[index.intValue()], reductions);
                }
                lrStateToMapSymbolToReduction.put(state, result);
            }
            return result;
        }
        
        
        // tags
        private String[] tags = null;
        private MultiMap tagToTaggedStates = null;
        public String[] tags() {
            if (tags == null) {
                Set tmp = new TreeSet();
                tagToTaggedStates = new MultiMap();
                LRState[] lrStates = lrStates();
                for (int i = 0; i < lrStates.length; i++) {
                    LRState lrState = lrStates[i];
                    if (lrState.tag() != null) {
                        String tag = lrState.tag().getImage();
                        tmp.add(tag);
                        tagToTaggedStates.add(tag, lrState);
                    }
                }
                tags = (String[]) tmp.toArray(new String[tmp.size()]);
            }
            return tags;
        }
        private Map tagToIndexNumber = null;
        public int getIndexOfTag(String tag) {
            if (tagToIndexNumber == null) {
                tagToIndexNumber = new HashMap();
                String[] tags = tags();
                for (int i = 0; i < tags.length; i++)
                    tagToIndexNumber.put(tags[i], new Integer(i));
            }
            Integer result = (Integer) tagToIndexNumber.get(tag);
            return result.intValue();
        }
        public Set getTaggedStates(String tag) {
            if (tagToTaggedStates == null) {
                tags();
            }
            return tagToTaggedStates.getValueSet(tag);
        }
        
        // types
        
        private Set types = null;
        public Set types() {
            if (types == null) {
                types = new LinkedHashSet(typeSystem.getMapNameToType().values());
            }
            return types;
        }
        
        private Set userDefinedTypes = null;
        public Set userDefinedTypes() {
            if (userDefinedTypes == null) {
                userDefinedTypes = new LinkedHashSet();
                Iterator it2 = types().iterator();
                while (it2.hasNext()) {
                    Type type = (Type) it2.next();
                    if (type instanceof UserDefinedType)
                        userDefinedTypes.add(type);
                }
            }
            return userDefinedTypes;
        }
        
    }
    
}
