
package jp.gr.java_conf.koto.notavacc.dfa;

import java.util.*;
import jp.gr.java_conf.koto.notavacc.*;
import jp.gr.java_conf.koto.notavacc.util.*;
import jp.gr.java_conf.koto.notavacc.lrg.*;
import jp.gr.java_conf.koto.notavacc.parser.Parser.*;
import jp.gr.java_conf.koto.notavacc.parser.Original.*;
import jp.gr.java_conf.koto.notavacc.parser.Parser.Root;

public class DFAGenerator extends Module {
    public DFAGenerator(Environment env) {
        super(env);
    }
    
    private Root root;
    public DeterministicFiniteAutomaton generate(final Root root) {
        this.root = root;
        
        logger.verbose(environment.processingFile, "generating DFA.");
        
        rejectSelfEmbedding(root.subtokenDefinitions());
        if (logger.hasError())
            logger.fatal();
        
        final MultiMap definitionToInitialState = new MultiMap();
        {
            Iterator it1 = root.tokenDefinitions().iterator();
            while (it1.hasNext()) {
                TokenDefinition def = (TokenDefinition) it1.next();
                definitionToInitialState.add(def, makeInitialState(def.expression()));
            }
            
            Set targetExpressions = new LinkedHashSet();
            Iterator it11 = root.aliasDefinitions().iterator();
            while (it11.hasNext()) {
                AliasDefinition def = (AliasDefinition) it11.next();
                targetExpressions.add(def.expression());
            }
            Iterator it12 = root.typeDefinitions().iterator();
            while (it12.hasNext()) {
                TypeDefinition def = (TypeDefinition) it12.next();
                if (def.abstractKeyword() == null)
                    targetExpressions.add(def.expression());
            }
            
            final Set done = new HashSet();
            Iterator it13 = targetExpressions.iterator();
            while (it13.hasNext()) {
                Expression expression = (Expression) it13.next();
                expression.accept(new Visitor() {
                    public void visit(Node node) {
                        if (node instanceof StringExpression) {
                            StringExpression exp = (StringExpression) node;
                            StringLiteral string = exp.string();
                            String value = root.getValue(string);
                            if (done.add(value)) {
                                StringState state = new StringState(value);
                                definitionToInitialState.add(exp, state);
                            }
                        }
                    }
                });
            }
        }
        WrapState initialState = new WrapState(definitionToInitialState);
        Map stringToNamedTerminal = new LinkedHashMap();
        Map stateToHitSymbol = getMapStateToHitSymbol(initialState, stringToNamedTerminal);
        
        logger.verbose(environment.processingFile, "generated.  {0} states.", new Integer(stateToHitSymbol.size()));
        logger.verbose(environment.processingFile, "building the minimal form.");
        
        DFA result = buildMinimalForm(initialState, stateToHitSymbol, stringToNamedTerminal);
        
        logger.verbose(environment.processingFile, "built.  {0} states.", new Integer(result.states().size()));
        
        if (logger.verboseLevel() >= Logger.LEVEL_VERBOSEST) {
            logger.verbosest(environment.processingFile, "----DFA States Report----");
            logger.verbosest("    initial state = {0}", result.initialState());
            Iterator it9 = result.states().iterator();
            while (it9.hasNext()) {
                DFAState state = (DFAState) it9.next();
                logger.verbosest("-- {0} --", state);
                if (state.hitSymbol != null)
                    logger.verbosest("hit {0}", state.hitSymbol);
                Iterator it10 = state.mapInputToNextStates.entrySet().iterator();
                while (it10.hasNext()) {
                    Map.Entry entry = (Map.Entry) it10.next();
                    DFAInput input = (DFAInput) entry.getKey();
                    DFAState nextState = (DFAState) entry.getValue();
                    logger.verbosest("    {0} ==> x{1}", input, nextState);
                }
            }
        }
        
        return result;
    }
    
    private static class DFAState implements DeterministicFiniteAutomaton.DFAState {
        private Terminal hitSymbol;
        public SortedMap mapInputToNextStates = new TreeMap();
        public DFAState(Terminal hitSymbol) {
            this.hitSymbol = hitSymbol;
        }
        public Terminal hitSymbol() {
            return hitSymbol;
        }
        public SortedMap getMapInputToNextStates() {
            return mapInputToNextStates;
        }
        
        private static int numberOfInstance = 0;
        public final int idNumber = numberOfInstance++;
        public String toString() {
            return "DFAState" + idNumber;
        }
    }
    private static class DFA extends DeterministicFiniteAutomaton {
        private Root root;
        private DFAState initialState;
        private Map stringToNamedTerminal;
        public DFA(Root root, DFAState initialState, Map stringToNamedTerminal) {
            this.root = root;
            this.initialState = initialState;
            this.stringToNamedTerminal = stringToNamedTerminal;
        }
        public DFAState initialState() {
            return initialState;
        }
        public Terminal getTerminal(StringExpression exp) {
            String str = root.getValue(exp.string());
            Terminal result = (Terminal) stringToNamedTerminal.get(str);
            if (result == null) {
                result = new StringTerminal(str);
            }
            return result;
        }
    }
    
    private void rejectSelfEmbedding(List definitions) {
        Set reportedDefinitions = new HashSet();
        Iterator it0 = definitions.iterator();
        while (it0.hasNext()) {
            SubtokenDefinition def = (SubtokenDefinition) it0.next();
            rejectSelfEmbedding(def, TreeStackSet.EMPTY_SET, reportedDefinitions);
        }
    }
    private void rejectSelfEmbedding(SubtokenDefinition definition, final TreeStackSet openedDefinitions, final Set reportedDefinitions) {
        definition.accept(new Visitor() {
            public void visit(Node node) {
                if (node instanceof IdentifierTokenExpression) {
                    IdentifierTokenExpression ite = (IdentifierTokenExpression) node;
                    SubtokenDefinition def = root.getDefinition(ite.name());
                    TreeStackSet openedDefs = TreeStackSet.push(openedDefinitions, def);
                    if (openedDefs.size() == openedDefinitions.size()) {
                        if (reportedDefinitions.add(def)) {
                            logger.error("The terminal {0} is defined using itself {1}.", def.identifier(), ite.name().identifier());
                        }
                    }
                    
                    rejectSelfEmbedding(def, openedDefs, reportedDefinitions);
                }
            }
        });
    }
    
    
    public static final SortedMap EMPTY_SORTED_MAP = Collections.unmodifiableSortedMap(new TreeMap());
    public static SortedMap singletonSortedMap(Object key, Object value) {
        SortedMap result = new TreeMap();
        result.put(key, value);
        return result;
    }
    public static MultiMap singletonMultiMap(Object key, Collection values) {
        MultiMap result = new MultiMap();
        result.put(key, new LinkedHashSet(values));
        return result;
    }
    
    private static abstract class NondeterministicState {
        private long hashCode = Long.MIN_VALUE;
        public final int hashCode() {
            if (hashCode == Long.MIN_VALUE) {
                hashCode = hashCodeOnce();
                if (hashCode == Long.MIN_VALUE) {
                    hashCode = Long.MIN_VALUE + 1;
                }
            }
            return (int) hashCode;
        }
        public abstract int hashCodeOnce();
        public abstract boolean equals(Object rhs);
        
        public abstract boolean isFinal();
        public abstract SortedMap/*<DFAInput,Set<NondeterministicState>>*/ getMapInputToNextStates();
        public Set/*<NondeterministicState>*/ getEpsilonMovedStates() {
            return Collections.EMPTY_SET;
        }
        
        public final Set/*<NondeterministicState>*/ getEpsilonClosure() {
            Set result = new LinkedHashSet();
            LinkedList undone = new LinkedList();
            undone.add(this);
            while (!undone.isEmpty()) {
                NondeterministicState state = (NondeterministicState) undone.removeFirst();
                if (result.add(state))
                    undone.addAll(state.getEpsilonMovedStates());
            }
            return result;
        }
    }
    
    private static abstract class DeterministicState extends NondeterministicState {
        private SortedMap mapInputToNextState = null;
        public final SortedMap getMapInputToNextState() {
            if (mapInputToNextState == null)
                mapInputToNextState = getMapInputToNextStateOnce();
            return mapInputToNextState;
        }
        
        public abstract SortedMap/*<DFAInput,DeterministicState>*/ getMapInputToNextStateOnce();
        
        private SortedMap mapInputToNextStates = null;
        public final SortedMap/*<DFAInput,Set<NondeterministicState>>*/ getMapInputToNextStates() {
            if (mapInputToNextStates == null) {
                mapInputToNextStates = new TreeMap(getMapInputToNextState());
                Iterator it = mapInputToNextStates.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry entry = (Map.Entry) it.next();
                    DeterministicState state = (DeterministicState) entry.getValue();
                    entry.setValue(Collections.singleton((NondeterministicState) state));
                }
            }
            return mapInputToNextStates;
        }
        
        public final Set getEpsilonMovedStates() {
            return Collections.EMPTY_SET;
        }
    }
    
    private static abstract class SingletonState extends DeterministicState {
        protected SingletonState() {
        }
        
        public int hashCodeOnce() {
            return getClass().getName().hashCode();
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            return getClass().equals(rhs.getClass());
        }
        
        public abstract boolean isFinal();
        public abstract SortedMap getMapInputToNextStateOnce();
    }
    private static final class FinalState extends SingletonState {
        public static final FinalState INSTANCE = new FinalState();
        
        private FinalState() {
        }
        
        public boolean isFinal() {
            return true;
        }
        
        public SortedMap getMapInputToNextStateOnce() {
            return EMPTY_SORTED_MAP;
        }
    }
    private static final class MatchAnyState extends SingletonState {
        public static final MatchAnyState INSTANCE = new MatchAnyState();
        
        private MatchAnyState() {
        }
        
        public boolean isFinal() {
            return true;
        }
        
        public SortedMap getMapInputToNextStateOnce() {
            return singletonSortedMap(new DFAInput(Character.MIN_VALUE, Character.MAX_VALUE), this);
        }
    }
    private static final class DeadState extends SingletonState {
        public static final DeadState INSTANCE = new DeadState();
        
        private DeadState() {
        }
        
        public boolean isFinal() {
            return false;
        }
        
        public SortedMap getMapInputToNextStateOnce() {
            return singletonSortedMap(new DFAInput(Character.MIN_VALUE, Character.MAX_VALUE), this);
        }
    }
    
    private static DeterministicState toDeterministicState(NondeterministicState ns) {
        if (ns instanceof DeterministicState)
            return (DeterministicState) ns;
        return new WrapState(ns);
    }
    
    private static class WrapState extends DeterministicState {
        private final MultiMap labelToStates;
        public WrapState(NondeterministicState initialState) {
            this(Collections.singleton(initialState));
        }
        public WrapState(Collection/*<NondeterministicState>*/ initialStates) {
            this(singletonMultiMap(null, initialStates));
        }
        public WrapState(MultiMap/*<Object,NondeterministicState>*/ labelToStates) {
            this.labelToStates = doEpsilonMove(labelToStates);
        }
        
        public Set labels() {
            Set result = labelToStates.keySet();
            assert (result = Collections.unmodifiableSet(result)) != null;
            return result;
        }
        
        public int hashCodeOnce() {
            return getClass().getName().hashCode() + labelToStates.hashCode();
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (this.getClass().equals(rhs.getClass()))
                return equals((WrapState) rhs);
            return false;
        }
        public boolean equals(WrapState rhs) {
            return labelToStates.equals(rhs.labelToStates);
        }
        
        private static MultiMap doEpsilonMove(MultiMap labelToStates) {
            MultiMap result = new MultiMap();
            Iterator outer = labelToStates.entrySet().iterator();
            while (outer.hasNext()) {
                Map.Entry entry = (Map.Entry) outer.next();
                Object label = entry.getKey();
                Set states = (Set) entry.getValue();
                Set closure = new LinkedHashSet();
                Iterator inner = states.iterator();
                while (inner.hasNext()) {
                    NondeterministicState state = (NondeterministicState) inner.next();
                    closure.addAll(state.getEpsilonClosure());
                }
                result.put(label, closure);
            }
            return result;
        }
        
        public boolean isFinal() {
            Iterator it = labels().iterator();
            while (it.hasNext()) {
                Object label = it.next();
                if (isFinal(label))
                    return true;
            }
            return false;
        }
        
        private Map labelToFlagFinal = new LinkedHashMap();
        protected boolean isFinal(Object label) {
            Boolean result = (Boolean) labelToFlagFinal.get(label);
            if (result == null) {
                result = Boolean.FALSE;
                Set states = (Set) labelToStates.get(label);
                if (states != null) {
                    Iterator it = states.iterator();
                    while (it.hasNext()) {
                        NondeterministicState state = (NondeterministicState) it.next();
                        if (state.isFinal()) {
                            result = Boolean.TRUE;
                            break;
                        }
                    }
                }
                labelToFlagFinal.put(label, result);
            }
            return result.booleanValue();
        }
        
        protected WrapState create(MultiMap labelToStates) {
            assert getClass().equals(WrapState.class);
            return new WrapState(labelToStates);
        }
        
        private static class Note {
            public final Object label;
            public final NondeterministicState state;
            public final List mapEntriesInputToNextStates;
            public Note(Object label, NondeterministicState state) {
                this.label = label;
                this.state = state;
                mapEntriesInputToNextStates = new LinkedList(state.getMapInputToNextStates().entrySet());
            }
        }
        public SortedMap getMapInputToNextStateOnce() {
            SortedMap result = new TreeMap();
            MultiMap labelToNotes = new MultiMap();
            {
                Iterator outer = labelToStates.entrySet().iterator();
                while (outer.hasNext()) {
                    Map.Entry entry = (Map.Entry) outer.next();
                    Object label = entry.getKey();
                    Set states = (Set) entry.getValue();
                    
                    Set notes = new LinkedHashSet();
                    Iterator inner = states.iterator();
                    while (inner.hasNext()) {
                        NondeterministicState state = (NondeterministicState) inner.next();
                        notes.add(new Note(label, state));
                    }
                    
                    labelToNotes.put(label, notes);
                }
            }
            
            int lower = Character.MIN_VALUE;
            while (lower <= Character.MAX_VALUE) {
                int upper = Character.MAX_VALUE;
                MultiMap labelToNextInnerStates = new MultiMap();
                
                Iterator outer = labelToNotes.entrySet().iterator();
                while (outer.hasNext()) {
                    Map.Entry entry = (Map.Entry) outer.next();
                    Object label = entry.getKey();
                    Set notes = (Set) entry.getValue();
                    
                    Set nextInnerStates = new LinkedHashSet();
                    Iterator inner = notes.iterator();
                    while (inner.hasNext()) {
                        Note note = (Note) inner.next();
                        
                        Iterator innest = note.mapEntriesInputToNextStates.iterator();
                        while (innest.hasNext()) {
                            Map.Entry e = (Map.Entry) innest.next();
                            DFAInput input = (DFAInput) e.getKey();
                            Set nextStates = (Set) e.getValue();
                            
                            if (input.lower > lower) {
                                upper = Math.min(upper, input.lower - 1);
                                break;
                            } else if (input.upper < lower) {
                                innest.remove();
                            } else {
                                assert input.lower <= lower && lower <= input.upper;
                                upper = Math.min(upper, input.upper);
                                nextInnerStates.addAll(nextStates);
                                break;
                            }
                        }
                    }
                    
                    if (!nextInnerStates.isEmpty())
                        labelToNextInnerStates.put(label, nextInnerStates);
                }
                
                if (!labelToNextInnerStates.isEmpty()) {
                    WrapState state = create(labelToNextInnerStates);
                    result.put(new DFAInput(lower, upper), state);
                }
                lower = upper + 1;
            }
            
            return result;
        }
    }
    
    private static class IntersectionState extends WrapState {
        private final Set labels;
        public IntersectionState(Set/*<NondeterministicState>*/ substates) {
            super(initialize(substates));
            labels = labels();
        }
        private static MultiMap initialize(Set/*<NondeterministicState>*/ substates) {
            MultiMap labelToStates = new MultiMap();
            int i = 0;
            Iterator it = substates.iterator();
            while (it.hasNext()) {
                NondeterministicState substate = (NondeterministicState) it.next();
                labelToStates.add(new Integer(i++), substate);
            }
            return labelToStates;
        }
        private IntersectionState(Set labels, MultiMap labelToStates) {
            super(labelToStates);
            this.labels = labels;
        }
        
        public boolean equals(WrapState rhs) {
            return equals((IntersectionState) rhs);
        }
        public boolean equals(IntersectionState rhs) {
            return super.equals((WrapState) rhs) && labels.equals(rhs.labels);
        }
        
        public boolean isFinal() {
            Iterator it = labels.iterator();
            while (it.hasNext()) {
                Object label = it.next();
                if (!isFinal(label))
                    return false;
            }
            return true;
        }
        
        protected WrapState create(MultiMap labelToStates) {
            assert getClass().equals(IntersectionState.class);
            return new IntersectionState(labels, labelToStates);
        }
    }
    
    private static class ComplementaryState extends DeterministicState {
        private final DeterministicState substate;
        public ComplementaryState(NondeterministicState substate) {
            this(toDeterministicState(substate));
        }
        public ComplementaryState(DeterministicState substate) {
            this.substate = substate;
        }
        
        public int hashCodeOnce() {
            return ~substate.hashCode();
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (rhs instanceof ComplementaryState)
                return equals((ComplementaryState) rhs);
            return false;
        }
        public boolean equals(ComplementaryState rhs) {
            return substate.equals(rhs.substate);
        }
        
        public boolean isFinal() {
            return !substate.isFinal();
        }
        
        public SortedMap/*<DFAInput,DeterministicState>*/ getMapInputToNextStateOnce() {
            SortedMap result = new TreeMap();
            int lower = Character.MIN_VALUE;
            Iterator it = substate.getMapInputToNextState().entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                DFAInput input = (DFAInput) entry.getKey();
                DeterministicState state = (DeterministicState) entry.getValue();
                if (lower < input.lower)
                    result.put(new DFAInput(lower, input.lower - 1), MatchAnyState.INSTANCE);
                result.put(input, new ComplementaryState(state));
                lower = input.upper + 1;
            }
            if (lower <= Character.MAX_VALUE)
                result.put(new DFAInput(lower, Character.MAX_VALUE), MatchAnyState.INSTANCE);
            return result;
        }
    }
    
    private class SequentialState extends NondeterministicState {
        private final List expressions;
        private final int index;
        private final DeterministicState substate;
        public SequentialState(List expressions) {
            this(expressions, 0);
        }
        public SequentialState(List expressions, int index) {
            if (expressions instanceof RandomAccess)
                this.expressions = expressions;
            else
                this.expressions = new ArrayList(expressions);
            this.index = index;
            
            substate = toDeterministicState(makeInitialState((TokenExpression) expressions.get(index)));
        }
        public SequentialState(List expressions, int index, DeterministicState substate) {
            this.expressions = expressions;
            this.index = index;
            this.substate = substate;
        }
        
        public int hashCodeOnce() {
            return getClass().getName().hashCode() + expressions.hashCode() + index + substate.hashCode();
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (getClass().equals(rhs.getClass()))
                return equals((SequentialState) rhs);
            return false;
        }
        public boolean equals(SequentialState rhs) {
            return expressions.equals(rhs.expressions) && index == rhs.index && substate.equals(rhs.substate);
        }
        
        public boolean isFinal() {
            return false;
        }
        
        public SortedMap/*<DFAInput,Set<NondeterministicState>>*/ getMapInputToNextStates() {
            SortedMap result = new TreeMap();
            Map map = substate.getMapInputToNextState();
            Iterator it = map.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                DFAInput input = (DFAInput) entry.getKey();
                DeterministicState state = (DeterministicState) entry.getValue();
                result.put(input, Collections.singleton(new SequentialState(expressions, index, state)));
            }
            return result;
        }
        
        public Set/*<NondeterministicState>*/ getEpsilonMovedStates() {
            if (substate.isFinal()) {
                NondeterministicState state;
                if (index + 1 < expressions.size()) {
                    state = new SequentialState(expressions, index + 1);
                } else {
                    state = FinalState.INSTANCE;
                }
                return Collections.singleton(state);
            } else {
                return Collections.EMPTY_SET;
            }
        }
    }
    
    private static final class PlusState extends NondeterministicState {
        private final DeterministicState initialSubstate;
        private final DeterministicState substate;  // null if the final state.
        public PlusState(NondeterministicState substate) {
            this(toDeterministicState(substate));
        }
        private PlusState(DeterministicState initialSubstate) {
            this(initialSubstate, initialSubstate);
        }
        private PlusState(DeterministicState initialSubstate, DeterministicState substate) {
            this.initialSubstate = initialSubstate;
            this.substate = substate;
        }
        
        
        public int hashCodeOnce() {
            return getClass().getName().hashCode() + initialSubstate.hashCode() + (substate == null ? 0 : substate.hashCode());
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (getClass().equals(rhs.getClass()))
                return equals((PlusState) rhs);
            return false;
        }
        public boolean equals(PlusState rhs) {
            return initialSubstate.equals(rhs.initialSubstate) && (substate == null ? substate == rhs.substate : substate.equals(rhs.substate));
        }
        
        public boolean isFinal() {
            return substate == null;
        }
        
        private SortedMap mapInputToNextStates = null;
        public SortedMap/*<DFAInput,Set<NondeterministicState>>*/ getMapInputToNextStates() {
            if (mapInputToNextStates == null) {
                mapInputToNextStates = new TreeMap();
                if (substate != null) {
                    Iterator it = substate.getMapInputToNextState().entrySet().iterator();
                    while (it.hasNext()) {
                        Map.Entry entry = (Map.Entry) it.next();
                        DFAInput input = (DFAInput) entry.getKey();
                        DeterministicState nextSubstate = (DeterministicState) entry.getValue();
                        PlusState nextState = new PlusState(initialSubstate, nextSubstate);
                        mapInputToNextStates.put(input, Collections.singleton(nextState));
                    }
                }
            }
            return mapInputToNextStates;
        }
        
        public Set getEpsilonMovedStates() {
            if (substate == null) {
                return Collections.singleton(new PlusState(initialSubstate));
            } else if (substate.isFinal()) {
                return Collections.singleton(new PlusState(initialSubstate, null));
            }
            return Collections.EMPTY_SET;
        }
    }
    
    private static class StringState extends DeterministicState {
        private final String string;
        private final int index;
        public StringState(String string) {
            this(string, 0);
        }
        public StringState(String string, int index) {
            this.string = string;
            this.index = index;
        }
        
        public int hashCodeOnce() {
            return string.hashCode() + index;
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (rhs instanceof StringState)
                return equals((StringState) rhs);
            return false;
        }
        public boolean equals(StringState rhs) {
            return string.equals(rhs.string) && index == rhs.index;
        }
        
        public boolean isFinal() {
            return index >= string.length();
        }
        
        public SortedMap/*<DFAInput,DeterministicState>*/ getMapInputToNextStateOnce() {
            if (index >= string.length())
                return EMPTY_SORTED_MAP;
            char ch = string.charAt(index);
            return singletonSortedMap(new DFAInput(ch, ch), new StringState(string, index + 1));
        }
        
        public String toString() {
            return "StringState('" + string + "', " + index + ")";
        }
    }
    
    private static class CharacterState extends DeterministicState {
        private final int lower;
        private final int upper;
        public CharacterState(int lower, int upper) {
            assert Character.MIN_VALUE <= lower && lower <= Character.MAX_VALUE;
            assert Character.MIN_VALUE <= upper && upper <= Character.MAX_VALUE;
            this.lower = lower;
            this.upper = upper;
        }
        
        public int hashCodeOnce() {
            return lower << 16 | upper;
        }
        public boolean equals(Object rhs) {
            if (this == rhs)
                return true;
            if (rhs instanceof CharacterState)
                return equals((CharacterState) rhs);
            return false;
        }
        public boolean equals(CharacterState rhs) {
            return lower == rhs.lower && upper == rhs.upper;
        }
        
        public boolean isFinal() {
            return false;
        }
        
        public SortedMap getMapInputToNextStateOnce() {
            return singletonSortedMap(new DFAInput(lower, upper), FinalState.INSTANCE);
        }
    }
    
    /*
    private static abstract class SingleCharState extends DeterministicState {
        public int hashCodeOnce() {
            return getClass().getName().hashCode();
        }
        public boolean equals(Object rhs) {
            return getClass().equals(rhs.getClass());
        }
        
        public boolean isFinal() {
            return false;
        }
        
        protected abstract boolean accepts(char ch);
        private SortedMap map = null;
        public SortedMap getMapInputToNextStateOnce() {
            if (map == null) {
                map = new TreeMap();
                int start = -1;
                for (int i = Character.MIN_VALUE; i <= Character.MAX_VALUE + 1; i++) {
                    if (i <= Character.MAX_VALUE && accepts((char) i)) {
                        if (start == -1)
                            start = i;
                    } else {
                        if (start != -1) {
                            map.put(new DFAInput(start, i - 1), FinalState.INSTANCE);
                            start = -1;
                        }
                    }
                }
            }
            return map;
        }
    }
    private static class JavaIdentifierStartState extends SingleCharState {
        public static JavaIdentifierStartState INSTANCE = new JavaIdentifierStartState();
        protected boolean accepts(char ch) {
            return Character.isJavaIdentifierStart(ch);
        }
    }
    private static class JavaIdentifierPartState extends SingleCharState {
        public static JavaIdentifierPartState INSTANCE = new JavaIdentifierPartState();
        protected boolean accepts(char ch) {
            return Character.isJavaIdentifierPart(ch);
        }
    }
    */
    
    private NondeterministicState makeInitialState(TokenExpression expression) {
        if (expression instanceof SelectiveTokenExpression) {
            SelectiveTokenExpression exp = (SelectiveTokenExpression) expression;
            if (exp.operands().size() == 0) {
                return FinalState.INSTANCE;
            } else if (exp.operands().size() == 1) {
                return makeInitialState((TokenExpression) exp.operands().iterator().next());
            }
            Set/*<NondeterministicState>*/ initialStates = new LinkedHashSet();
            Iterator it = exp.operands().iterator();
            while (it.hasNext()) {
                TokenExpression sub = (TokenExpression) it.next();
                initialStates.add(makeInitialState(sub));
            }
            return new WrapState(initialStates);
        } else if (expression instanceof IntersectionTokenExpression) {
            Set/*<NondeterministicState>*/ substates = new LinkedHashSet();
            for (;;) {
                IntersectionTokenExpression exp = (IntersectionTokenExpression) expression;
                substates.add(makeInitialState(exp.rhs()));
                expression = exp.lhs();
                if (!(expression instanceof IntersectionTokenExpression)) {
                    substates.add(makeInitialState(expression));
                    break;
                }
            }
            return new IntersectionState(substates);
        } else if (expression instanceof SequentialTokenExpression) {
            SequentialTokenExpression exp = (SequentialTokenExpression) expression;
            if (exp.operands().size() == 0) {
                return FinalState.INSTANCE;
            } else if (exp.operands().size() == 1) {
                return makeInitialState((TokenExpression) exp.operands().iterator().next());
            }
            return new SequentialState(exp.operands());
        } else if (expression instanceof ComplementaryTokenExpression) {
            ComplementaryTokenExpression exp = (ComplementaryTokenExpression) expression;
            return new ComplementaryState(makeInitialState(exp.operand()));
        } else if (expression instanceof PlusTokenExpression) {
            PlusTokenExpression exp = (PlusTokenExpression) expression;
            return new PlusState(makeInitialState(exp.operand()));
        } else if (expression instanceof IdentifierTokenExpression) {
            IdentifierTokenExpression exp = (IdentifierTokenExpression) expression;
            TokenExpression te = root.getDefinition(exp.name()).expression();
            return makeInitialState(te);
        /*} else if (expression instanceof PreparedSubtokenExpression) {
            PreparedSubtokenExpression exp = (PreparedSubtokenExpression) expression;
            if (exp.name().getImage().equals("$JAVA_IDENTIFIER_START")) {
                return JavaIdentifierStartState.INSTANCE;
            } else {
                assert exp.name().getImage().equals("$JAVA_IDENTIFIER_PART");
                return JavaIdentifierPartState.INSTANCE;
            }*/
        } else if (expression instanceof StringTokenExpression) {
            StringTokenExpression exp = (StringTokenExpression) expression;
            StringLiteral sl = (StringLiteral) exp.string();
            return new StringState(root.getValue(sl));
        } else {
            assert (expression instanceof CharacterRangeTokenExpression);
            CharacterRangeTokenExpression exp = (CharacterRangeTokenExpression) expression;
            CharacterLiteral lower = (CharacterLiteral) exp.lower();
            CharacterLiteral upper = (CharacterLiteral) exp.upper();
            return new CharacterState(root.getValue(lower), root.getValue(upper));
        }
    }
    
    
    private Map getMapStateToHitSymbol(WrapState initialState, Map stringToNamedTerminal) {
        // reject zero-length tokens.
        Iterator it2 = initialState.labels().iterator();
        while (it2.hasNext()) {
            Object definition = (Object) it2.next();
            if (initialState.isFinal(definition)) {
                if (definition instanceof TokenDefinition) {
                    TokenDefinition def = (TokenDefinition) definition;
                    logger.error("The terminal {0} should not match the empty string.", def.identifier());
                } else {
                    StringExpression def = (StringExpression) definition;
                    logger.error("The terminal {0} should not be the empty string.", def.string().token());
                }
            }
        }
        
        // check acceptance.
        MultiMap definitionToFinalStates = new MultiMap();
        Map result = new LinkedHashMap();
        LinkedList undone = new LinkedList();
        undone.add(initialState);
        while (!undone.isEmpty()) {
            WrapState state = (WrapState) undone.removeFirst();
            if (!result.containsKey(state)) {
                result.put(state, null);
                Iterator it = state.labels().iterator();
                while (it.hasNext()) {
                    Object definition = it.next();
                    if (state.isFinal(definition))
                        definitionToFinalStates.add(definition, state);
                }
                
                undone.addAll(state.getMapInputToNextState().values());
            }
        }
        
        // bind string expressions to token definitions.
        {
            MultiMap finalStateSetToDefinitions = new MultiMap();
            Iterator it4 = definitionToFinalStates.entrySet().iterator();
            while (it4.hasNext()) {
                Map.Entry entry = (Map.Entry) it4.next();
                finalStateSetToDefinitions.add(entry.getValue(), entry.getKey());
            }
            Iterator it6 = finalStateSetToDefinitions.values().iterator();
            while (it6.hasNext()) {
                Set definitions = (Set) it6.next();
                if (definitions.size() == 2) {
                    TokenDefinition tokenDefinition;
                    StringExpression stringExpression;
                    
                    Iterator it = definitions.iterator();
                    Object d = it.next();
                    if (d instanceof TokenDefinition) {
                        tokenDefinition = (TokenDefinition) d;
                        stringExpression = (StringExpression) it.next();
                    } else {
                        stringExpression = (StringExpression) d;
                        tokenDefinition = (TokenDefinition) it.next();
                    }
                    String string = root.getValue(stringExpression.string());
                    NamedTerminal symbol = new NamedTerminal(root.getCanonicalName(tokenDefinition));
                    stringToNamedTerminal.put(string, symbol);
                }
            }
            
        }
        
        // bind hit symbols.
        Set errorReportedDefinitionSets = new HashSet();
        Iterator outer = result.entrySet().iterator();
        while (outer.hasNext()) {
            Map.Entry entry = (Map.Entry) outer.next();
            WrapState state = (WrapState) entry.getKey();
            if (!state.isFinal()) {
                entry.setValue(null);
            } else {
                Set definitions = new LinkedHashSet(state.labels());
                Iterator inner = definitions.iterator();
                while (inner.hasNext()) {
                    Object definition = inner.next();
                    if (!state.isFinal(definition)) {
                        inner.remove();
                    } else {
                        if (definition instanceof StringExpression) {
                            StringExpression exp = (StringExpression) definition;
                            String str = root.getValue(exp.string());
                            if (stringToNamedTerminal.containsKey(str))
                                inner.remove();
                        }
                    }
                }
                assert !definitions.isEmpty();
                if (definitions.size() > 1) {
                    if (errorReportedDefinitionSets.add(definitions)) {
                        if (!state.equals(initialState)) {
                            boolean first = true;
                            Iterator it3 = definitions.iterator();
                            while (it3.hasNext()) {
                                Object definition = (Object) it3.next();
                                Token token;
                                if (definition instanceof TokenDefinition) {
                                    TokenDefinition def = (TokenDefinition) definition;
                                    token = def.identifier();
                                } else {
                                    StringExpression def = (StringExpression) definition;
                                    token = def.string().token();
                                }
                                if (first) {
                                    logger.error("The followings match the same string.", token);
                                    first = false;
                                }
                                logger.error("        {0}", token);
                            }
                        }
                    }
                }
                Object definition = definitions.iterator().next();
                if (definition instanceof TokenDefinition) {
                    TokenDefinition def = (TokenDefinition) definition;
                    NamedTerminal hitSymbol = new NamedTerminal(root.getCanonicalName(def));
                    entry.setValue(hitSymbol);
                } else {
                    StringExpression def = (StringExpression) definition;
                    StringTerminal hitSymbol = new StringTerminal(root.getValue(def.string()));
                    entry.setValue(hitSymbol);
                }
            }
        }
        
        return result;
    }
    
    private DFA buildMinimalForm(DeterministicState initialState, Map stateToHitSymbol, Map stringToNamedTerminal) {
        // initialize
        List groups = new ArrayList(stateToHitSymbol.size());
        Map stateToGroupID = new HashMap();
        {
            Map hitSymbolToGroupID = new HashMap();
            hitSymbolToGroupID.put(null, new Integer(0));
            groups.add(new LinkedList(Collections.singleton(DeadState.INSTANCE)));
            stateToGroupID.put(DeadState.INSTANCE, new Integer(0));
            
            Iterator it5 = stateToHitSymbol.entrySet().iterator();
            while (it5.hasNext()) {
                Map.Entry entry = (Map.Entry) it5.next();
                DeterministicState state = (DeterministicState) entry.getKey();
                Terminal hitSymbol = (Terminal) entry.getValue();
                Integer groupID = (Integer) hitSymbolToGroupID.get(hitSymbol);
                if (groupID == null) {
                    groupID = new Integer(groups.size());
                    groups.add(new LinkedList());
                    hitSymbolToGroupID.put(hitSymbol, groupID);
                }
                List group = (List) groups.get(groupID.intValue());
                group.add(state);
                stateToGroupID.put(state, groupID);
            }
        }
        
        // loop
        {
            int lastMoodifiedIndex = -1;
            for (int index = 0;; index++) {
                if (index >= groups.size()) {
                    if (lastMoodifiedIndex == -1)
                        break;
                    index -= groups.size();
                }
                if (index == lastMoodifiedIndex)
                    break;
                
                List group = (List) groups.get(index);
                assert !group.isEmpty();
                if (group.size() <= 1) {
                    if (index == lastMoodifiedIndex)
                        break;
                    continue;
                }
                
                Map keyToStateList = new LinkedHashMap();
                Iterator it = group.iterator();
                while (it.hasNext()) {
                    DeterministicState state = (DeterministicState) it.next();
                    SortedMap key = new TreeMap();
                    int prevUpper = Character.MIN_VALUE - 1;
                    Iterator it7 = state.getMapInputToNextState().entrySet().iterator();
                    while (it7.hasNext()) {
                        Map.Entry entry = (Map.Entry) it7.next();
                        DFAInput input = (DFAInput) entry.getKey();
                        DeterministicState nextState = (DeterministicState) entry.getValue();
                        if (input.lower > prevUpper + 1)
                            key.put(new DFAInput(prevUpper + 1, input.lower - 1), stateToGroupID.get(DeadState.INSTANCE));
                        key.put(input, stateToGroupID.get(nextState));
                        prevUpper = input.upper;
                    }
                    if (prevUpper < Character.MAX_VALUE)
                        key.put(new DFAInput(prevUpper + 1, Character.MAX_VALUE), stateToGroupID.get(DeadState.INSTANCE));
                    key = minimizeMap(key);
                    List list = (List) keyToStateList.get(key);
                    if (list == null) {
                        list = new LinkedList();
                        keyToStateList.put(key, list);
                    }
                    list.add(state);
                }
                if (keyToStateList.size() <= 1) {
                    if (index == lastMoodifiedIndex)
                        break;
                    continue;
                }
                
                lastMoodifiedIndex = index;
                Iterator outer = keyToStateList.values().iterator();
                while (outer.hasNext()) {
                    List newGroup = (List) outer.next();
                    if (outer.hasNext()) {
                        Integer newID = new Integer(groups.size());
                        Iterator inner = newGroup.iterator();
                        while (inner.hasNext()) {
                            DeterministicState state = (DeterministicState) inner.next();
                            stateToGroupID.put(state, newID);
                        }
                        groups.add(newGroup);
                    } else {
                        groups.set(index, newGroup);
                    }
                }
            }
        }
        
        // finalize
        Integer deadIndexInteger = (Integer) stateToGroupID.get(DeadState.INSTANCE);
        int deadIndex = deadIndexInteger.intValue();
        DFAState[] states = new DFAState[groups.size()];
        for (int i = 0; i < states.length; i++) {
            if (i != deadIndex) {
                List group = (List) groups.get(i);
                DeterministicState state = (DeterministicState) group.get(0);
                states[i] = new DFAState((Terminal) stateToHitSymbol.get(state));
            }
        }
        for (int i = 0; i < states.length; i++) {
            if (i != deadIndex) {
                List group = (List) groups.get(i);
                DeterministicState state = (DeterministicState) group.get(0);
                DFAState newState = states[i];
                Iterator it8 = state.getMapInputToNextState().entrySet().iterator();
                while (it8.hasNext()) {
                    Map.Entry entry = (Map.Entry) it8.next();
                    DFAInput input = (DFAInput) entry.getKey();
                    DeterministicState nextState = (DeterministicState) entry.getValue();
                    Integer index = (Integer) stateToGroupID.get(nextState);
                    if (index.intValue() != deadIndex)
                        newState.mapInputToNextStates.put(input, states[index.intValue()]);
                }
                newState.mapInputToNextStates = minimizeMap(newState.mapInputToNextStates);
            }
        }
        
        Integer initialIndexInteger = (Integer) stateToGroupID.get(initialState);
        int initialIndex = initialIndexInteger.intValue();
        DFAState resultInitialState = states[initialIndex];
        if (resultInitialState == null)
            resultInitialState = new DFAState(null);
        return new DFA(root, resultInitialState, stringToNamedTerminal);
    }
    
    private SortedMap minimizeMap(SortedMap inputToObject) {
        if (inputToObject.size() <= 1)
            return inputToObject;
        
        SortedMap result = new TreeMap();
        Iterator it = inputToObject.entrySet().iterator();
        Object previousValue;
        int previousLower;
        int previousUpper;
        {
            Map.Entry entry = (Map.Entry) it.next();
            DFAInput input = (DFAInput) entry.getKey();
            previousValue = entry.getValue();
            previousLower = input.lower;
            previousUpper = input.upper;
        }
        
        while (it.hasNext()) {
            Map.Entry entry = (Map.Entry) it.next();
            DFAInput input = (DFAInput) entry.getKey();
            Object value = entry.getValue();
            
            if (input.lower > previousUpper + 1 || !(previousValue == null ? value == null : previousValue.equals(value))) {
                result.put(new DFAInput(previousLower, previousUpper), previousValue);
                previousValue = value;
                previousLower = input.lower;
                previousUpper = input.upper;
            } else {
                previousUpper = input.upper;
            }
        }
        result.put(new DFAInput(previousLower, previousUpper), previousValue);
        return result;
    }
    
}

