
package jp.gr.java_conf.koto.notavacc;

import java.util.*;
import java.io.*;
import jp.gr.java_conf.koto.util.*;
import jp.gr.java_conf.koto.io.*;
import jp.gr.java_conf.koto.notava.*;
import jp.gr.java_conf.koto.notavacc.parser.*;
import jp.gr.java_conf.koto.notavacc.dfa.*;
import jp.gr.java_conf.koto.notavacc.lrg.*;
import jp.gr.java_conf.koto.notavacc.types.*;
import jp.gr.java_conf.koto.notavacc.generator.*;

public class Main {
    private static final String NAME = Alphabet.J + Alphabet.A + Alphabet.V + Alphabet.A + Alphabet.C + Alphabet.C;
    private static final String COMMAND_NAME = Version.NAME;
    private static final String COPYRIGHT = "Copyright " + Alphabet.NORMAL_COPYRIGHT + " Koto 2001-2002.  Licensed under " + Version.LICENSE + ".";
    
    public static String getBaseName(File file) {
        String name = file.getName();
        int index = name.lastIndexOf(".");
        if (index > 0)
            name = name.substring(0, index);
        return name;
    }
    
    private static List getRelativePathItems(File base, File file) throws IOException {
        base = base.getCanonicalFile();
        File f = file.getCanonicalFile();
        
        LinkedList items = new LinkedList();
        do {
            items.addFirst(f.getName());
            f = f.getParentFile();
        } while (f != null && !f.equals(base));
        
        return items;
    }
    public static String getFileName(File file, boolean withoutParentPath) {
        if (withoutParentPath)
            return file.getName();
        
        try {
            File userDir = new File(System.getProperty("user.dir"));
            List items = getRelativePathItems(userDir, file);
            
            File f = null;
            Iterator it0 = items.iterator();
            while (it0.hasNext()) {
                String item = (String) it0.next();
                f = new File(f, item);
            }
            
            return f.getPath();
        } catch (IOException x) {
            return file.toString();
        }
    }
    
    public static File getReplacedPath(File file, String sourceDir, String destinationDir) throws IOException {
        return getReplacedPath(file, new File(sourceDir), new File(destinationDir));
    }
    public static File getReplacedPath(File file, File sourceDir, File destinationDir) throws IOException {
        File result = destinationDir;
        List items = getRelativePathItems(sourceDir, file);
        Iterator it1 = items.iterator();
        while (it1.hasNext()) {
            String item = (String) it1.next();
            result = new File(result, item);
        }
        return result;
    }
    
    public static void compile(Environment env, List files) throws Exception {
        boolean withoutParentPath = env.commandLine.countOf("x-without-parent-path") > 0;
        String sourceDir = env.commandLine.argumentOf("source-path");
        String destinationDir = env.commandLine.argumentOf("destination-path");
        
        double targetVersion = 1.4;
        try {
            String version = System.getProperty("java.vm.version");
            boolean priodProcessed = false;
            int i = 0;
            for (; i < version.length(); i++) {
                char ch = version.charAt(i);
                if (ch == '.') {
                    if (priodProcessed)
                        break;
                    priodProcessed = true;
                } else if (!('0' <= ch && ch <= '9'))
                    break;
            }
            targetVersion = Double.parseDouble(version.substring(0, i));
        } catch (NumberFormatException x) {
        }
        
        String targetVersionString = env.commandLine.argumentOf("target");
        if (targetVersionString == null) {
            env.logger.warn("--target missing.  The VM version ({0}) is used.", Double.toString(targetVersion));
        } else {
            try {
                targetVersion = Double.parseDouble(targetVersionString);
                if (targetVersion < 1.2) {
                    env.logger.warn("The target version {0} should be grater than or equal to 1.2.", Double.toString(targetVersion));
                }
            } catch (NumberFormatException x) {
                env.logger.warn("{0} should be the Java varsion of the output. e.g. 1.4", targetVersionString);
            }
        }
        
        {
            double[] supportedTargets = { 1.2, 1.3, 1.4 } ;
            int i = 0;
            for (; i < supportedTargets.length; i++) {
                if (targetVersion == supportedTargets[i])
                    break;
            }
            if (i >= supportedTargets.length) {
                StringBuffer buffer = new StringBuffer();
                for (i = 0; i < supportedTargets.length - 1; i++) {
                    buffer.append(supportedTargets[i]);
                    buffer.append(", ");
                }
                env.logger.warn("The target version {0} is not supported.  The supported versions are {1}and {2}.", Double.toString(targetVersion), buffer, Double.toString(supportedTargets[i]));
            }
        }
        
        Iterator it = files.iterator();
        while (it.hasNext()) {
            String fileName = (String) it.next();
            File sourceFile = new File(fileName);
            env.processingFile = sourceFile;
            File parentFile = sourceFile.getParentFile();
            
            File outputFile;
            if (destinationDir == null) {
                outputFile = new File(parentFile, getBaseName(sourceFile) + ".java");
            } else {
                if (sourceDir == null) {
                    outputFile = new File(destinationDir, getBaseName(sourceFile) + ".java");
                } else {
                    outputFile = getReplacedPath(new File(parentFile, getBaseName(sourceFile) + ".java"), sourceDir, destinationDir);
                }
            }
            if (outputFile.getParentFile() != null)
                outputFile.getParentFile().mkdirs();
            
            String sourceFileName = getFileName(sourceFile, withoutParentPath);
            String outputFileName = getFileName(outputFile, withoutParentPath);
            env.logger.inform("generating a parser from ''{0}'' into ''{1}''.", sourceFileName, outputFileName);
            
            Parser.Root root;
            try {
                root = (Parser.Root) new Parser(env.logger).parseRoot(sourceFileName, sourceFile, null, 8);
            } catch (Parser.ParseException x) {
                env.logger.fatal(x.getSourceName(), x.getLine(), "{0}", x.getLocalizedMessage());
                assert false;
                root = null;
            }
            
            DeterministicFiniteAutomaton dfa = new DFAGenerator(env).generate(root);
            LRTable lrTable = new LRTableGenerator(env).generate(root, dfa);
            TypeSystem typeSystem = new TypeSystemGenerator(env, targetVersion >= 1.5).generate(root, lrTable.getTypeNonterminals());
            boolean dryRun = env.commandLine.countOf("dry-run") > 0;
            boolean error = true;
            try {
                if (!env.logger.hasError()) {
                    new JavaCodeGenerator(env, root, dfa, lrTable, typeSystem, targetVersion).generate(outputFile, dryRun);
                    error = false;
                }
            } finally {
                if (error && !dryRun)
                    outputFile.delete();
            }
        }
    }
    
    public static Throwable process(String[] args) {
        return process(args, null);
    }
    public static Throwable process(String[] args, Writer writer) {
        CommandLineChecker clc = new CommandLineChecker(new CommandLineParameter[] {
            new CommandLineParameter('v',   "verbose",                  "increase verbosity."),
            new CommandLineParameter('q',   "quiet",                    "decrease verbosity."),
            new CommandLineParameter(       "source-path",      "dir",  "input from <dir>."),
            new CommandLineParameter('d',   "destination-path", "dir",  "output to <dir>."),
            new CommandLineParameter(       "target",           "type", "output target: version of java"),
            new CommandLineParameter(       "dry-run",                  "output no code."),
            new CommandLineParameter('X',   "experimental",             "use experimental algorithms.", true),
            new CommandLineParameter(       "debug",                    "debugging mode for compiler-compiler developers.", true),
            new CommandLineParameter(       "x-without-parent-path",    "print the file name without the parent path.", true),
            new CommandLineParameter(       "version",                  "show the version."),
            new CommandLineParameter('h',   "help",                     "display this help."),
        }, args);
        
        List files = clc.arguments();
        boolean withoutParentPath = clc.countOf("x-without-parent-path") > 0;
        int verboseLevel = Logger.LEVEL_INFORMATION + clc.countOf("verbose") - clc.countOf("quiet");
        boolean usage = clc.countOf("help") > 0 || (clc.countOf("version") <= 0 && files.isEmpty());
        boolean version = verboseLevel >= Logger.LEVEL_VERBOSE || usage || clc.countOf("version") > 0;
        Logger logger = new Logger(withoutParentPath, verboseLevel, writer);
        Environment env = new Environment(logger, clc);
        
        try {
            if (version) {
                logger.inform(NAME + " Version " + Version.VERSION + " (" + Version.DATE + ")");
                logger.inform(COPYRIGHT);
            }
            if (usage) {
                StringWriter stringWriter = new StringWriter();
                PrintWriter printWriter = new PrintWriter(stringWriter);
                printWriter.println("Usage: " + COMMAND_NAME + " [options] files...");
                clc.printOptionUsage(printWriter, verboseLevel >= Logger.LEVEL_VERBOSE);
                printWriter.close();
                assert !printWriter.checkError();
                logger.inform("{0}", stringWriter.toString());
            } else {
                compile(env, files);
            }
            if (logger.hasError())
                logger.fatal();
            return null;
        } catch (Logger.FatalErrorException x) {
            // already reported.
            return x;
        } catch (Throwable x) {
            if (x instanceof IOException)
                logger.error("I/O error: {0}", x.getLocalizedMessage());
            else
                logger.error("An exception: {0}", x.getLocalizedMessage());
            
            StringWriter stringWriter = new StringWriter();
            PrintWriter printWriter = new PrintWriter(stringWriter);
            x.printStackTrace(printWriter);
            printWriter.close();
            assert !printWriter.checkError();
            if (x instanceof IOException)
                logger.verbose("{0}", stringWriter.toString());
            else
                logger.inform("{0}", stringWriter.toString());
            return x;
        } finally {
            try {
                logger.close();
            } catch (IOException x) {
                System.err.println("error: cannot write a log (into stdout maybe).");
            }
        }
    }
    
    public static void main(String[] args) {
        if (process(args) == null)
            System.exit(0);
        else
            System.exit(1);
    }
    
}

