
import java.util.*;
import java.io.*;
import parser.*;
import parser.Syntax.*;

public class Main {
    private final Map nameToHTML;
    public String get(String name) {
        return (String) nameToHTML.get(name);
    }
    
    public Main(File input) throws Exception {
        this(input, null);
    }
    public Main(File input, PrintWriter output) throws Exception {
        nameToHTML = new LinkedHashMap();
        
        Root root = new Parser().parseRoot(input);
        root.accept(new Visitor() {
            public void visit(Node node) {
                if (node instanceof TypeDefinition) {
                    TypeDefinition def = (TypeDefinition) node;
                    if (def.abstractKeyword() == null)
                        pick(def, def.expression());
                } else if (node instanceof AliasDefinition) {
                    AliasDefinition def = (AliasDefinition) node;
                    pick(def, def.expression());
                }
            }
            public void pick(Definition def, Expression expression) {
                String name = def.identifier().getImage();
                String html = getHTML(name, expression);
                nameToHTML.put(name, html);
            }
        });
        
        if (output != null) {
            output.println("<!doctype html public \"-//W3C//DTD HTML 4.01 Transitional//EN\">");
            output.println("<html>");
            output.println("<head>");
            output.println("<meta http-equiv='Content-Script-Type' content='text/javascript'>");
            output.println("<title>" + encode(input.getName()) + "</title>");
            output.println("<script type='text/javascript'><!--");
            output.println(""
                    + "var oldDecoration = \"none\";\n"
                    + "function getTarget(element) {\n"
                    + "  if (document.all) {\n"
                    + "    var id = null;\n"
                    + "    if (element.href != null) {\n"
                    + "      id = element.href;\n"
                    + "      var index = id.indexOf(\"#\");\n"
                    + "      if (index < 0)\n"
                    + "        id = null;\n"
                    + "      else\n"
                    + "        id = id.substring(index + 1);\n"
                    + "    }\n"
                    + "    if (id != null) {\n"
                    + "      var target = document.all.item(id, 0);\n"
                    + "      if (target != null) {\n"
                    + "        return target;\n"
                    + "      }\n"
                    + "    }\n"
                    + "  }\n"
                    + "  return null;\n"
                    + "}\n"
                    + "function highlightTarget(element) {\n"
                    + "  var target = getTarget(element);\n"
                    + "  if (target != null) {\n"
                    + "    oldDecoration = target.style.textDecoration;\n"
                    + "    target.style.textDecoration = \"underline\";\n"
                    + "  }\n"
                    + "}\n"
                    + "function unhighlightTarget(element) {\n"
                    + "  var target = getTarget(element);\n"
                    + "  if (target != null) {\n"
                    + "    target.style.textDecoration = oldDecoration;\n"
                    + "  }\n"
                    + "}\n"
            );
            output.println("//--></script>");
            output.println("</head>");
            output.println("<body>");
            output.println("<pre>");
            
            Iterator it1 = nameToHTML.values().iterator();
            while (it1.hasNext()) {
                String html = (String) it1.next();
                output.println(html);
            }
            
            output.println("</pre>");
            output.println("</body></html>");
            output.flush();
        }
    }
    
    public static String getHTML(String name, Expression expression) {
        StringBuffer result = new StringBuffer();
        result.append("<a id='syntax:" + encode(name) + "' name='syntax:" + encode(name) + "' class='syntax'>");
        result.append(encode(name));
        result.append("</a>");
        result.append(" ");
        result.append("::=");
        result.append(" ");
        getHTML(result, 0, expression);
        return result.toString();
    }
    
    private static void getHTML(StringBuffer buffer, int level, Expression expression) {
        if (expression instanceof IdentifierExpression) {
            IdentifierExpression exp = (IdentifierExpression) expression;
            String name = exp.name().identifier().getImage();
            buffer.append("<a href='#syntax:" + encode(name) + "' class='syntax' onMouseOver='highlightTarget(this)' onMouseOut='unhighlightTarget(this)'>");
            buffer.append(encode(name));
            buffer.append("</a>");
            return;
        }
        
        if (expression instanceof StringExpression) {
            StringExpression exp = (StringExpression) expression;
            buffer.append(encode(exp.string().token().getImage()));
            return;
        }
        
        if (expression instanceof PlusExpression) {
            PlusExpression exp = (PlusExpression) expression;
            
            if (level > 2)
                buffer.append("( ");
            
            getHTML(buffer, 2, exp.operand());
            buffer.append(encode("+"));
            
            if (level > 2)
                buffer.append(" )");
            return;
        }
        
        if (expression instanceof SelectiveExpression) {
            SelectiveExpression exp = (SelectiveExpression) expression;
            boolean hitsEmpty = false;
            List nonemptyOperans = new ArrayList(exp.operands().size());
            Iterator it2 = exp.operands().iterator();
            while (it2.hasNext()) {
                Expression operand = (Expression) it2.next();
                boolean e = false;
                if (operand instanceof SequentialExpression) {
                    if (((SequentialExpression) operand).operands().isEmpty())
                        e = true;
                }
                if (e) {
                    hitsEmpty = true;
                } else {
                    nonemptyOperans.add(operand);
                }
            }
            if (hitsEmpty) {
                if (nonemptyOperans.isEmpty())
                    return;
                if (nonemptyOperans.size() == 1) {
                    Expression operand = (Expression) nonemptyOperans.get(0);
                    if (operand instanceof PlusExpression) {
                        PlusExpression pex = (PlusExpression) operand;
                        if (level > 2)
                            buffer.append("( ");
                        
                        getHTML(buffer, 2, pex.operand());
                        buffer.append(encode("*"));
                        
                        if (level > 2)
                            buffer.append(" )");
                    } else {
                        getHTML(buffer, 2, operand);
                        buffer.append(encode("?"));
                    }
                    return;
                }
                
                buffer.append("( ");
                Iterator it0 = nonemptyOperans.iterator();
                while (it0.hasNext()) {
                    Expression operand = (Expression) it0.next();
                    getHTML(buffer, 1, operand);
                    if (it0.hasNext())
                        buffer.append(" ");
                }
                buffer.append(" )");
                buffer.append(encode("?"));
                return;
            }
        }
        
        if (expression instanceof SequentialExpression) {
            SequentialExpression exp = (SequentialExpression) expression;
            if (exp.operands().isEmpty())
                return;
            if (exp.operands().size() == 1) {
                getHTML(buffer, level, (Expression) exp.operands().get(0));
                return;
            }
            
            if (level > 1)
                buffer.append("( ");
            Iterator it0 = exp.operands().iterator();
            while (it0.hasNext()) {
                Expression operand = (Expression) it0.next();
                getHTML(buffer, 1, operand);
                if (it0.hasNext())
                    buffer.append(" ");
            }
            if (level > 1)
                buffer.append(" )");
            return;
        }
        
        if (expression instanceof SelectiveExpression) {
            SelectiveExpression exp = (SelectiveExpression) expression;
            if (exp.operands().isEmpty())
                return;
            if (exp.operands().size() == 1) {
                getHTML(buffer, level, (Expression) exp.operands().get(0));
                return;
            }
            
            if (level > 0)
                buffer.append("( ");
            Iterator it0 = exp.operands().iterator();
            while (it0.hasNext()) {
                Expression operand = (Expression) it0.next();
                getHTML(buffer, 0, operand);
                if (it0.hasNext())
                    buffer.append(" | ");
            }
            if (level > 0)
                buffer.append(" )");
            return;
        }
        
        throw new AssertionError();
    }
    
    private static String encode(String image) {
        StringBuffer result = new StringBuffer();
        for (int i = 0; i < image.length(); i++) {
            char ch = image.charAt(i);
            if ('a' <= ch && ch <= 'z' || 'A' <= ch && ch <= 'Z' || '0' <= ch && ch <= '9') {
                result.append(ch);
            } else if (ch == '_') {
                result.append(ch);
            } else {
                result.append("&#" + ((int) ch) + ";");
            }
        }
        return result.toString();
    }
    
    public static void main(String[] args) {
        try {
            if (args.length != 1) {
                System.out.println("usage: notavacc2html <file>");
            } else {
                PrintWriter output = new PrintWriter(new BufferedWriter(new OutputStreamWriter(System.out)));
                try {
                    new Main(new File(args[0]), output);
                } finally {
                    output.close();
                    if (output.checkError())
                        throw new IOException();
                }
            }
        } catch (Exception x) {
            x.printStackTrace();
        }
    }
}

